<?php

namespace App\Imports;

use App\Models\Employee;
use App\Models\Shift;
use App\Models\WorkSchedule;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\SkipsFailures;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Carbon\Carbon;

class WorkSchedulesImport implements ToModel, WithHeadingRow, SkipsOnFailure, WithChunkReading
{
    use SkipsFailures;

    protected $startDate;
    protected $endDate;
    public $processedRows = [];

    public function __construct($startDate, $endDate = null)
    {
        $this->startDate = $startDate;
        $this->endDate = $endDate;
    }

    // public function headingRow(): int
    // {
    //     return 2; // headers on row 2
    // }

    public function chunkSize(): int
    {
        return 100; // process 100 rows at a time
    }

    public function model(array $row)
    {
        if (empty(array_filter($row))) {
            return null;
        }

        try {
            return DB::transaction(function () use ($row) {

                // 🔹 default row data
                $rowData = $row;

                // 🔹 Find Employee
                // Try finding employee by name (case-insensitive)
                $employee = Employee::whereRaw('LOWER(first_name) = ?', [strtolower(trim($row['first_name'] ?? ''))])
                    ->whereRaw('LOWER(last_name) = ?', [strtolower(trim($row['last_name'] ?? ''))])
                    ->first();

                // If BIO No. column is present in Excel, also check EmployeeClientChannel.dtr_id
                if ($employee && !empty($row['BIO No.'])) {
                    $assign = \App\Models\EmployeeClientChannel::where('employee_id', $employee->id)
                        ->where('dtr_id', trim($row['BIO No.']))
                        ->first();

                    if (!$assign) {
                        // Not found in assignemployeee table
                        $employee = null;
                    }
                }


                if (!$employee) {
                    $this->processedRows[] = ['remark' => "❌ Employee not found", 'row' => $rowData];
                    return null;
                }

                if (empty($row['sched'])) {
                    $this->processedRows[] = ['remark' => "❌ No schedule provided", 'row' => $rowData];
                    return null;
                }

                [$shiftStart, $shiftEnd] = $this->parseTimeRange($row['sched']);
                if (!$shiftStart || !$shiftEnd) {
                    $this->processedRows[] = ['remark' => "❌ Invalid schedule format", 'row' => $rowData];
                    return null;
                }

                [$breakOut, $breakIn] = $this->parseTimeRange($row['lunch_break'] ?? null);

                $shift = Shift::where('shift_name', $row['sched'])
                    ->where('shift_start', $shiftStart)
                    ->where('shift_end', $shiftEnd)
                    ->where('break_out', $breakOut)
                    ->where('break_in', $breakIn)
                    ->first();

                if (!$shift) {
                    $shift = Shift::create([
                        'shift_name' => $row['sched'],
                        'shift_start' => $shiftStart,
                        'shift_end' => $shiftEnd,
                        'break_out' => $breakOut,
                        'break_in' => $breakIn,
                        'status' => 'Approved',
                    ]);
                }

                $exists = WorkSchedule::where('employee_id', $employee->id)
                    ->where('shift_id', $shift->id)
                    ->where('start_date', $this->startDate)
                    ->where('end_date', $this->endDate)
                    ->exists();

                if ($exists) {
                    $this->processedRows[] = ['remark' => "⚠️ Duplicate schedule", 'row' => $rowData];
                    return null;
                }

                $schedule = WorkSchedule::create([
                    'employee_id' => $employee->id,
                    'shift_id' => $shift->id,
                    'start_date' => $this->startDate,
                    'end_date' => $this->endDate,
                    'status' => $row['status'] ?? 'active',
                ]);

                if (!$schedule) {
                    $this->processedRows[] = ['remark' => "❌ Failed to save schedule", 'row' => $rowData];
                    return null;
                }

                $this->processedRows[] = ['remark' => "✅ Imported successfully", 'row' => $rowData];
                return $schedule;
            });
        } catch (\Throwable $e) {
            $this->processedRows[] = [
                'remark' => "❌ Exception: " . $e->getMessage(),
                'row' => $row,
            ];
            return null;
        }
    }

    private function parseTimeRange(?string $timeRange, bool $isLunch = false): array
    {
        if (empty($timeRange) || !str_contains($timeRange, '-')) {
            if (is_numeric($timeRange)) {
                try {
                    $carbonDate = \Carbon\Carbon::instance(
                        \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($timeRange)
                    );
                    return [$carbonDate->format('H:i:s'), $carbonDate->format('H:i:s')];
                } catch (\Exception $e) {
                    return [null, null];
                }
            }
            return [null, null];
        }

        [$start, $end] = array_map('trim', explode('-', $timeRange));

        $normalizeTime = function ($time) {
            if (preg_match('/am|pm/i', $time)) {
                return $time; // already specified
            }
            $hour = (int) explode(':', $time)[0];
            if ($hour === 12) {
                return $time . 'pm'; // 12 always pm unless explicitly written
            }
            return ($hour >= 1 && $hour <= 11) ? $time . 'am' : $time . 'pm';
        };

        $start = $normalizeTime($start);
        $end = $normalizeTime($end);

        try {
            return [
                Carbon::parse($start)->format('H:i:s'),
                Carbon::parse($end)->format('H:i:s')
            ];
        } catch (\Exception $e) {
            return [null, null];
        }
    }

}
