<?php

namespace App\Imports;

use App\Models\Ptsr;
use App\Models\PtsrTrx;
use App\Models\EmployeeClientChannel;
use App\Models\TrxCode;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithCalculatedFormulas;
use Illuminate\Support\Facades\Log;

class PayrollSummaryImport implements ToModel, WithHeadingRow, WithCalculatedFormulas
{
    protected $startDate;
    protected $endDate;
    protected $department_id;

    public $failedRows = [];

    public function __construct($startDate, $endDate, $department_id)
    {
        $this->startDate = $startDate;
        $this->endDate = $endDate;
        $this->department_id = $department_id;
    }

    public function model(array $row)
    {
        // Normalize keys
        $normalizedRow = array_change_key_case(array_map('trim', $row), CASE_LOWER);

        $lastName = $normalizedRow['last_name'];
        $firstName = $normalizedRow['first_name'];
        $middleName = $normalizedRow['middle_name'] ?? '';

        $rowStatus = [
            'row_data' => $row,
            'status' => 'success',
            'errors' => [],
            'trx_codes' => []
        ];

        /**
         * 🔍 Step 1: Check directly in EmployeeClientChannel
         */
        $assignedEmployees = EmployeeClientChannel::where('department_id', $this->department_id)
            ->where('status', 1)
            ->whereHas('employee', function ($q) use ($lastName, $firstName, $middleName) {
                $q->where('last_name', 'like', "%{$lastName}%")
                    ->where('first_name', 'like', "%{$firstName}%");

                if (!empty($middleName)) {
                    $q->where('middle_name', 'like', "%{$middleName}%");
                } else {
                    $q->where(function ($q) {
                        $q->whereNull('middle_name')
                            ->orWhere('middle_name', '');
                    });
                }
            })
            ->with('employee')
            ->get();

        if ($assignedEmployees->isEmpty()) {
            $errorMessage = "Employee not found or not assigned to department {$this->department_id}: {$lastName}, {$firstName}" . ($middleName ? " {$middleName}" : '');
            Log::error($errorMessage);
            $rowStatus['status'] = 'failed';
            $rowStatus['errors'][] = $errorMessage;
            $this->failedRows[] = $rowStatus;
            return null;
        }

        if ($assignedEmployees->count() > 1) {
            $errorMessage = "Multiple employees found in department {$this->department_id} with name {$lastName}, {$firstName}" . ($middleName ? " {$middleName}" : '');
            Log::error($errorMessage);
            $rowStatus['status'] = 'failed';
            $rowStatus['errors'][] = $errorMessage;
            $this->failedRows[] = $rowStatus;
            return null;
        }

        // ✅ Only one employee assignment found
        $employee = $assignedEmployees->first()->employee;

        /**
         * 🔹 Step 2: Map columns
         */
        $mapping = [
            'number_of_days' => 'total_of_days',
            'under_time' => 'lates_undertime',
            'total' => 'o',
            'ot' => 'ot_125%',
            'duty' => 'drd',
            'special_premium' => 'special_holiday',
            'adj' => 'transpo'
        ];

        $ptsrData = [
            'employee_id' => $employee->id,
            'department_id' => $this->department_id,
            'start_date' => $this->startDate,
            'end_date' => $this->endDate,
            'status' => 'summary',
        ];

        foreach ($mapping as $key => $header) {
            $columnKey = strtolower(str_replace(' ', '_', $header));
            if (isset($normalizedRow[$columnKey])) {
                $cleanValue = preg_replace('/[^0-9.]/', '', $normalizedRow[$columnKey]);
                $ptsrData[$key] = is_numeric($cleanValue) ? (float) $cleanValue : 0;
            } else {
                $ptsrData[$key] = 0;
            }
        }

        // Save PTSR
        $ptsr = Ptsr::create($ptsrData);

        /**
         * 🔹 Step 3: Process all row columns for TrxCode mapping
         */
        foreach ($row as $header => $value) {
            if (empty($value)) {
                continue;
            }

            $value = trim(mb_convert_encoding($value, 'UTF-8', 'auto'));
            $value = str_replace("\xC2\xA0", ' ', $value);

            $cleanValue = preg_replace('/[^0-9.]/', '', $value);
            $amount = is_numeric($cleanValue) ? (float) $cleanValue : 0;

            $normalizedHeader = strtolower(trim(preg_replace('/\s+/', ' ', $header)));

            $trxCode = TrxCode::whereRaw("LOWER(TRIM(REPLACE(description, '  ', ' '))) LIKE ?", ["%{$normalizedHeader}%"])
                ->first();

            if ($trxCode) {
                PtsrTrx::create([
                    'ptsr_id' => $ptsr->id,
                    'code_id' => $trxCode->id,
                    'amount' => $amount,
                ]);

                $rowStatus['trx_codes'][] = [
                    'header' => $header,
                    'status' => 'exists',
                    'message' => "TrxCode '{$header}' found and saved."
                ];
            } else {
                $missingTrxMessage = "TrxCode Not Found for Header: '{$header}'";
                Log::error($missingTrxMessage);

                $rowStatus['trx_codes'][] = [
                    'header' => $header,
                    'status' => 'not_found',
                    'message' => $missingTrxMessage
                ];

                $rowStatus['status'] = 'warning';
            }
        }

        $this->failedRows[] = $rowStatus;

        return $ptsr;
    }

    public function getFailedRows()
    {
        return array_map(function ($row) {
            return [
                'row_data' => $row['row_data'],
                'status' => $row['status'],
                'errors' => !empty($row['errors']) ? implode(', ', $row['errors']) : 'No errors',
                'trx_codes' => array_map(function ($trx) {
                    return "{$trx['header']}: {$trx['message']}";
                }, $row['trx_codes']),
            ];
        }, $this->failedRows);
    }
}
