<?php

// namespace App\Imports;

// use App\Models\Employee;
// use Maatwebsite\Excel\Concerns\ToCollection;
// use Illuminate\Support\Collection;
// use App\Models\EmployeeClientChannel;
// use Carbon\Carbon;
// use Exception;
// use Illuminate\Support\Facades\DB;

// class AttendanceImport2 implements ToCollection
// {
//     const BATCH_SIZE = 100000;
//     public $failedRows = [];
//     public $departmentId;

//     public function __construct($departmentId)
//     {
//         $this->departmentId = $departmentId;
//     }

//     public function collection(Collection $rows)
//     {
//         $requiredHeaders = ['EnNo', 'IN/OUT', 'DateTime'];

//         // Get the header row and map column positions
//         $headerRow = $rows->first()->mapWithKeys(function ($value, $index) {
//             return [$value => $index];
//         });

//         // Ensure all required headers exist in the file
//         foreach ($requiredHeaders as $header) {
//             if (!isset($headerRow[$header])) {
//                 throw new Exception("Missing required column: $header");
//             }
//         }

//         // Remove header row
//         $rows->shift();

//         $attendanceData = [];
//         $results = [];

//         // Preserve original row indexes
//         $indexedRows = $rows->map(function ($row, $index) use ($headerRow) {
//             return [
//                 'row' => $index + 2, // Adjusting for the header row
//                 'EnNo' => $row[$headerRow['EnNo']],
//                 'IN/OUT' => $row[$headerRow['IN/OUT']],
//                 'DateTime' => $row[$headerRow['DateTime']]
//             ];
//         });

//         // Group data by Employee and Date
//         $groupedData = $indexedRows->groupBy(function ($row) {
//             $date = $this->parseDateTime($row['DateTime']);
//             return $date ? $row['EnNo'] . '-' . $date->format('Y-m-d') : null;
//         });

//         foreach ($groupedData as $key => $entries) {
//             if (!$key) {
//                 foreach ($entries as $entry) {
//                     $results[$entry['row']] = [
//                         'line' => $entry['row'],
//                         'EnNo' => $entry['EnNo'],
//                         'IN/OUT' => $entry['IN/OUT'],
//                         'DateTime' => $this->parseDateTime($entry['DateTime']) ? $this->parseDateTime($entry['DateTime'])->format('n/j/Y g:i:s A') : $entry['DateTime'],
//                         'error' => 'Invalid DateTime format.',
//                     ];
//                 }
//                 continue;
//             }

//             $employeeCode = $entries->first()['EnNo'];
//             $date = $this->parseDateTime($entries->first()['DateTime'])->format('Y-m-d');

//             $employeeChannels = EmployeeClientChannel::with('employee')
//                 ->where('status', '1')
//                 ->where('dtr_id', $employeeCode)
//                 ->where('department_id', $this->departmentId)
//                 ->get();

//             if ($employeeChannels->isEmpty()) {
//                 foreach ($entries as $entry) {
//                     $results[$entry['row']] = [
//                         'line' => $entry['row'],
//                         'EnNo' => $entry['EnNo'],
//                         'IN/OUT' => $entry['IN/OUT'],
//                         'DateTime' => $this->parseDateTime($entry['DateTime']) ? $this->parseDateTime($entry['DateTime'])->format('n/j/Y g:i:s A') : $entry['DateTime'],
//                         'error' => "No active employee channel found for EnNo: $employeeCode in department {$this->departmentId}.",
//                     ];
//                 }
//                 continue;
//             }

//             $timeIn = null;
//             $timeOut = null;

//             foreach ($entries as $entry) {
//                 $timestamp = $this->parseDateTime($entry['DateTime']);

//                 if (in_array($entry['IN/OUT'], ['S', '5', '0']) && !$timeIn) {
//                     $timeIn = $timestamp;
//                 }

//                 if (in_array($entry['IN/OUT'], ['E', '6', '1'])) {
//                     $timeOut = $timestamp;
//                 }
//             }

//             foreach ($employeeChannels as $channel) {
//                 $exists = DB::table('attendances')
//                     ->where('employee_id', $channel->employee_id)
//                     ->where('date', $date)
//                     ->where('department_id', $this->departmentId)
//                     ->exists();

//                 if ($exists) {
//                     foreach ($entries as $entry) {
//                         $results[$entry['row']] = [
//                             'line' => $entry['row'],
//                             'EnNo' => $entry['EnNo'],
//                             'IN/OUT' => $entry['IN/OUT'],
//                             'DateTime' => $this->parseDateTime($entry['DateTime']) ? $this->parseDateTime($entry['DateTime'])->format('n/j/Y g:i:s A') : $entry['DateTime'],
//                             'error' => "Duplicate attendance record exists for EnNo: {$channel->employee->last_name} {$channel->employee->first_name} on {$date}.",
//                         ];
//                     }
//                     continue;
//                 }

//                 $attendanceData[] = [
//                     'employee_id' => $channel->employee_id,
//                     'department_id' => $this->departmentId,
//                     'date' => $date,
//                     'time_in' => $timeIn,
//                     'time_out' => $timeOut,
//                     'notes' => 'Imported data',
//                     'created_at' => now(),
//                     'updated_at' => now(),
//                 ];

//                 foreach ($entries as $entry) {
//                     $results[$entry['row']] = [
//                         'line' => $entry['row'],
//                         'EnNo' => $entry['EnNo'],
//                         'IN/OUT' => $entry['IN/OUT'],
//                         'DateTime' => $this->parseDateTime($entry['DateTime']) ? $this->parseDateTime($entry['DateTime'])->format('n/j/Y g:i:s A') : $entry['DateTime'],
//                         'error' => "Inserted successfully, Employee Name: {$channel->employee->last_name} {$channel->employee->first_name}.",
//                     ];
//                 }
//             }
//         }

//         if (!empty($attendanceData)) {
//             $this->insertAttendanceBatch($attendanceData);
//         }

//         ksort($results);
//         $this->failedRows = array_values($results);
//     }



//     private function insertAttendanceBatch($attendanceData)
//     {
//         if (!empty($attendanceData)) {
//             DB::table('attendances')->insert($attendanceData);
//         }
//     }

//     private function parseDateTime($dateTime)
//     {
//         $dateTime = trim($dateTime);
//         if (is_numeric($dateTime)) {
//             return Carbon::createFromFormat('Y-m-d H:i', gmdate('Y-m-d H:i', ($dateTime - 25569) * 86400));
//         }

//         $formats = [
//             'n/j/Y H:i',
//             'm/d/Y H:i',
//             'Y-m-d H:i',
//             'n/j/Y g:i',
//             'm/d/Y g:i',
//             'Y-m-d g:i',
//             'Y-m-d H:i:s',
//             'm/d/Y H:i:s',
//             'n/j/Y H:i:s',
//             'n/j/Y g:i:s',
//             'm/d/Y H:i A',
//             'n/j/Y g:i A',
//             'Y-m-d g:i A',
//             'Y-m-d h:i A',
//             'm/d/Y h:i A',
//             'n/j/Y h:i A',
//             'Y-m-d h:i A',
//             'Y/m/d H:i',
//             'Y.m.d H:i',
//             'd/m/Y H:i',
//             'd.m.Y H:i',
//             'd/m/Y g:i A',
//             'd.m.Y g:i A',
//             'd/m/Y h:i A',
//             'd.m.Y h:i A',
//             'Y-m-d\TH:i:s\Z',
//             'D, d M Y H:i:s O',
//             'D, d M Y H:i:s T',
//             'Y-m-d\TH:i:s',
//             'Y/m/d H:i:s',
//             'Y-m-d H:i:s.u',
//         ];

//         foreach ($formats as $format) {
//             try {
//                 return Carbon::createFromFormat($format, $dateTime);
//             } catch (Exception $e) {
//                 continue;
//             }
//         }

//         return null;
//     }

namespace App\Imports;

use App\Models\Employee;

use App\Models\Attendance;
use App\Models\EmployeeClientChannel;
use App\Models\WorkSchedule;
use App\Models\Shift;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\ToCollection;

class AttendanceImport2 implements ToCollection
{
    const BATCH_SIZE = 1000;
    public $failedRows = [];
    public $departmentId;

    public function __construct($departmentId)
    {
        $this->departmentId = $departmentId;
    }

    public function collection(Collection $rows)
    {
        $requiredHeaders = ['EnNo', 'DateTime'];

        $headerRow = $rows->first()->mapWithKeys(fn($value, $index) => [$value => $index]);

        foreach ($requiredHeaders as $header) {
            if (!isset($headerRow[$header])) {
                throw new Exception("Missing required column: $header");
            }
        }

        $rows->shift();

        $attendanceData = [];
        $results = [];

        $indexedRows = $rows->map(fn($row, $index) => [
            'row' => $index + 2,
            'EnNo' => $row[$headerRow['EnNo']],
            'DateTime' => $this->parseDateTime($row[$headerRow['DateTime']])
        ]);

        $groupedData = $indexedRows->groupBy(fn($row) => $row['DateTime'] ? $row['EnNo'] . '-' . $row['DateTime']->format('Y-m-d') : null);

        foreach ($groupedData as $key => $entries) {
            if (!$key) {
                foreach ($entries as $entry) {
                    $results[$entry['row']] = ['line' => $entry['row'], 'EnNo' => $entry['EnNo'], 'DateTime' => $entry['DateTime'], 'error' => 'Invalid DateTime format.'];
                }
                continue;
            }

            $employeeCode = $entries->first()['EnNo'];
            $date = $entries->first()['DateTime']->format('Y-m-d');

            // Employee and work schedule logic
            $employee = EmployeeClientChannel::where('dtr_id', $employeeCode)
                ->where('department_id', $this->departmentId)
                ->where('status', '1')
                ->first();

            if (!$employee) {
                foreach ($entries as $entry) {
                    $results[$entry['row']] = ['line' => $entry['row'], 'EnNo' => $entry['EnNo'], 'DateTime' => $entry['DateTime'], 'error' => "No active employee found for EnNo: $employeeCode."];
                }
                continue;
            }



            $exists = DB::table('attendances')
                ->where('employee_id', $employee->employee_id)
                ->where('date', $date)
                ->where('department_id', $this->departmentId)
                ->first();

            if ($exists) {
                foreach ($entries as $entry) {
                    $results[$entry['row']] = ['line' => $entry['row'], 'EnNo' => $entry['EnNo'], 'DateTime' => $entry['DateTime'], 'error' => "Duplicate attendance record exists for EnNo: {$employee->employee->last_name} {$employee->employee->first_name} on {$date}."];
                }
                continue;
            }










            // Retrieve work schedule and shift data for the employee
            $workSchedule = WorkSchedule::where('employee_id', $employee->employee_id)
                ->where('start_date', '<=', $date)
                ->where('end_date', '>=', $date)
                ->first();

            if ($workSchedule) {
                $shift = Shift::find($workSchedule->shift_id);

                if ($shift) {
                    // Use the shift's times as references
                    $shiftStart = Carbon::parse($date . ' ' . $shift->shift_start);
                    $shiftEnd = Carbon::parse($date . ' ' . $shift->shift_end);
                    $breakOutTarget = Carbon::parse($date . ' ' . $shift->break_out);
                    $breakInTarget = Carbon::parse($date . ' ' . $shift->break_in);
                } else {
                    // Fallback if no shift is found for the work schedule
                    $shiftStart = Carbon::parse($date . ' ' . $employee->entry_time);
                    $shiftEnd = Carbon::parse($date . ' ' . $employee->exit_time);
                    $breakOutTarget = Carbon::parse($date . ' 12:00:00');
                    $breakInTarget = Carbon::parse($date . ' 13:00:00');
                }
            } else {
                // Fallback to default shift times if no work schedule is found
                $shiftStart = Carbon::parse($date . ' ' . $employee->entry_time);
                $shiftEnd = Carbon::parse($date . ' ' . $employee->exit_time);
                $breakOutTarget = Carbon::parse($date . ' 12:00:00');
                $breakInTarget = Carbon::parse($date . ' 13:00:00');
            }

            // Filter the records for this employee and date from the Excel data
            $employeeRecords = collect($entries)->pluck('DateTime');

            // Use the shift times as references for nearest times
            $timeIn = $this->findNearestTime($employeeRecords, $shiftStart);  // Nearest to shift start
            $breakOut = $this->findNearestTime($employeeRecords, $breakOutTarget);  // Nearest to break-out
            $breakIn = $this->findNearestTime($employeeRecords, $breakInTarget);  // Nearest to break-in
            $timeOut = $this->findNearestTime($employeeRecords, $shiftEnd);  // Nearest to shift end

            // Insert data only if attendance data is found
            $attendanceData[] = [
                'employee_id' => $employee->employee_id,
                'department_id' => $this->departmentId,
                'date' => $date,
                'time_in' => $timeIn ?? null,
                'break_out' => $breakOut ?? null,
                'break_in' => $breakIn ?? null,
                'time_out' => $timeOut ?? null,
                'notes' => 'Imported data',
                'created_at' => now(),
                'updated_at' => now(),
            ];

            foreach ($entries as $entry) {
                $results[$entry['row']] = ['line' => $entry['row'], 'EnNo' => $entry['EnNo'], 'DateTime' => $entry['DateTime'], 'error' => "Inserted successfully, Employee DTR ID: {$employee->dtr_id} {$employee->employee->last_name} {$employee->employee->first_name}."];
            }
        }

        if (!empty($attendanceData)) {
            $chunks = array_chunk($attendanceData, self::BATCH_SIZE);
            foreach ($chunks as $chunk) {
                DB::table('attendances')->insert($chunk);
            }
        }

        ksort($results);
        $this->failedRows = array_values($results);
    }

    private function findNearestTime($employeeRecords, $shiftTime)
    {
        $nearest = null;
        $smallestDiff = null;
        $threshold = 120; // 2 hours in minutes

        foreach ($employeeRecords as $record) {
            $time = Carbon::parse($record);
            $diff = abs($time->diffInMinutes($shiftTime));

            if (($smallestDiff === null || $diff < $smallestDiff) && $diff <= $threshold) {
                $nearest = $time;
                $smallestDiff = $diff;
            }
        }

        return $nearest; // Will be null if no record is within 2 hours
    }

    private function parseDateTime($dateTime)
    {
        $dateTime = trim($dateTime);
        if (is_numeric($dateTime)) {
            return Carbon::createFromFormat('Y-m-d H:i', gmdate('Y-m-d H:i', ($dateTime - 25569) * 86400));
        }

        $formats = [
            'n/j/Y H:i',
            'm/d/Y H:i',
            'Y-m-d H:i',
            'n/j/Y g:i',
            'm/d/Y g:i',
            'Y-m-d g:i',
            'Y-m-d H:i:s',
            'm/d/Y H:i:s',
            'n/j/Y H:i:s',
            'n/j/Y g:i:s',
            'm/d/Y H:i A',
            'n/j/Y g:i A',
            'Y-m-d g:i A',
            'Y-m-d h:i A',
            'm/d/Y h:i A',
            'n/j/Y h:i A',
            'Y-m-d h:i A',
            'Y/m/d H:i',
            'Y.m.d H:i',
            'd/m/Y H:i',
            'd.m.Y H:i',
            'd/m/Y g:i A',
            'd.m.Y g:i A',
            'd/m/Y h:i A',
            'd.m.Y h:i A',
            'Y-m-d\TH:i:s\Z',
            'D, d M Y H:i:s O',
            'D, d M Y H:i:s T',
            'Y-m-d\TH:i:s',
            'Y/m/d H:i:s',
            'Y-m-d H:i:s.u',
        ];

        foreach ($formats as $format) {
            try {
                return Carbon::createFromFormat($format, $dateTime);
            } catch (Exception $e) {
                continue;
            }
        }

        return null;
    }
}
