<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Stores;
use App\Models\User;
use Yajra\DataTables\DataTables;
use Illuminate\Support\Facades\Hash;
use App\Models\Department;

class StoreController extends Controller
{
    /**
     * Display a listing of the stores.
     */
    public function index(Request $request)
    {
        $data = Stores::with('user')->get();

        if ($request->ajax()) {
            return DataTables::of($data)
                ->addColumn('action', function ($row) {
                    $viewBtn = '<a href="' . route('storeassignedepartment', $row->id) . '" 
                   class="flex items-center justify-center w-10 h-10 text-white bg-blue-500 rounded-md">
                    <i class="text-xl bx bx-street-view"></i>
                </a>';

                    $editBtn = '<button class="flex items-center justify-center w-10 h-10 text-white bg-yellow-500 rounded-md edit-store" data-id="' . $row->id . '">
                                    <i class="text-xl bx bx-edit-alt"></i>
                                </button>';

                    $deleteBtn = '<button class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md delete-file" data-id="' . $row->id . '">
                                    <i class="text-xl bx bx-trash-alt"></i>
                                </button>';

                    return '<div class="flex items-center justify-center gap-2">' . $viewBtn . $editBtn . $deleteBtn . '</div>';
                })
                ->rawColumns(['action'])
                ->make(true);
        }

        return view('Admin.Store.index', compact('data'));
    }

    /**
     * Show the form for creating a new store.
     */
    // public function assignstore(Request $request)
    // {
    //     $data = Stores::with('user')->get();
    //     $department = Department::get();
    //     if ($request->ajax()) {
    //         return DataTables::of($data)
    //             ->addColumn('action', function ($row) {
    //                 $viewBtn = '<button class="flex items-center justify-center w-10 h-10 text-white bg-blue-500 rounded-md edit-store" data-id="' . $row->id . '">
    //                                 <i class="text-xl bx bx-street-view"></i>
    //                             </button>';

    //                 $editBtn = '<button class="flex items-center justify-center w-10 h-10 text-white bg-yellow-500 rounded-md edit-store" data-id="' . $row->id . '">
    //                                 <i class="text-xl bx bx-edit-alt"></i>
    //                             </button>';

    //                 $deleteBtn = '<button class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md delete-file" data-id="' . $row->id . '">
    //                                 <i class="text-xl bx bx-trash-alt"></i>
    //                             </button>';

    //                 return '<div class="flex items-center justify-center gap-2">' . $viewBtn . $editBtn . $deleteBtn . '</div>';
    //             })
    //             ->rawColumns(['action'])
    //             ->make(true);
    //     }

    //     return view('Admin.Store.storeassign', compact('data', 'department'));
    // }
    /**
     * Store a newly created store in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:3|confirmed',
            'name' => 'required|string|max:255',
        ]);

        // Create user
        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'role_id' => 23, // fixed role
        ]);

        // Create store
        Stores::create([
            'user_id' => $user->id,
            'name' => $validated['name'],
        ]);

        return response()->json(['success' => true, 'message' => 'Store created successfully']);
    }

    /**
     * Display the specified store.
     */
    public function show($id)
    {
        $store = Stores::with('user')->findOrFail($id);
        return response()->json($store);
    }

    /**
     * Show the form for editing the specified store.
     */
    public function edit($id)
    {
        $store = Stores::with('user')->findOrFail($id);

        return response()->json([
            'id' => $store->id,
            'name' => $store->name,
            'email' => $store->user->email ?? '',
        ]);
    }

    /**
     * Update the specified store in storage.
     */
    public function update(Request $request, $id)
    {
        $store = Stores::findOrFail($id);
        $user = $store->user;

        $validated = $request->validate([
            'email' => 'required|string|email|max:255|unique:users,email,' . $user->id,
            'name' => 'required|string|max:255',
            'password' => 'nullable|string|min:3',
        ]);

        // Update user
        $user->update([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => $request->filled('password') ? Hash::make($request->password) : $user->password,
        ]);

        // Update store
        $store->update([
            'name' => $validated['name'],
        ]);

        return response()->json(['success' => true, 'message' => 'Store updated successfully']);
    }

    /**
     * Remove the specified store from storage.
     */
    public function destroy($id)
    {
        $store = Stores::findOrFail($id);
        $user = $store->user;

        // Delete store first
        $store->delete();

        // Delete linked user
        if ($user) {
            $user->delete();
        }

        return response()->json(['success' => true, 'message' => 'Store deleted successfully']);
    }
}
