<?php

namespace App\Http\Controllers;

use App\Models\Shift;
use App\Models\Assign;
use App\Models\AssignCompany;
use App\Models\AssignCompanyDepartment;
use App\Models\Employee;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Carbon\Carbon;

use App\Models\WorkSchedule;

class ShiftController extends Controller
{
    // Get all shifts
    public function index(Request $request)
    {
        $user = Auth::user();

        // Get the agency IDs assigned to the user
        $agencyIds = $user->viewable_agency_ids;

        // Query Employees under agencies assigned to this user
        $query = Employee::with('channelclient');

        $query->whereHas('channelclient', function ($query) {
            $query->whereNotNull('department_id');
        });

        if (!empty($agencyIds)) {

            $query->whereHas('channelclient.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id'))
                ->pluck('department_id')
                ->unique();

            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereHas('channelclient', function ($q) use ($restrictedDepartmentIds) {
                    $q->whereIn('department_id', $restrictedDepartmentIds);
                });
            }
        }
        $employee = $query->get();





        if ($request->ajax()) {
            $data = Shift::get();

            return DataTables::of($data)
                ->addColumn('action', function ($row) {
                    return '
            <div class="flex items-center justify-center gap-2">
              <button onclick="openEditModal(' . $row->id . ')" class="flex items-center justify-center w-10 h-10 text-white bg-blue-600 rounded-md">
                        <i class="text-xl bx bx-edit"></i>
                    </button>
                <button onclick="openDeleteModal(' . $row->id . ')" class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md">
                    <i class="text-xl bx bx-trash-alt"></i>
                </button>
            </div>';
                })
                ->rawColumns(['action'])
                ->make(true);
        }
        $shift = Shift::all();



        return view('Admin.Employee.shedule.shifts', compact('employee', 'shift'));
    }
    public function pendingschedule(Request $request)
    {


        $user = Auth::user();

        // Get the agency IDs the user is assigned to
        $agencyIds = $user->viewable_agency_ids;
        $query = WorkSchedule::with('shift', 'employee.channelclient.department');

        if (!empty($agencyIds)) {

            $query->whereHas('employee.channelclient.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id'))
                ->pluck('department_id')
                ->unique();

            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereHas('employee.channelclient', function ($q) use ($restrictedDepartmentIds) {
                    $q->whereIn('department_id', $restrictedDepartmentIds);
                });
            }
        }

        if ($request->ajax()) {
            $data = $query->orderByRaw('status IS NULL DESC, status ASC')->get();

            if ($data->isEmpty()) {
                return response()->json([]);
            }

            return DataTables::of($data)
                ->addColumn('approve', function ($row) {
                    // Check if the status is null before rendering the button
                    if ($row->status === null) {
                        return '
                    <div class="flex items-center justify-center gap-2">
                        <button onclick="openApproveModal(' . $row->id . ')" class="flex items-center justify-center w-10 h-10 text-white bg-green-600 rounded-md">
                            <i class="text-xl bx bx-check"></i>
                        </button>
                    </div>';
                    }

                    // If status is not null, return an empty string to hide the button
                    return '';
                })


                ->addColumn('action', function ($row) {
                    return '
                <div class="flex items-center justify-center gap-2">
                  <a href="' . url('admin/pre-profile/' . $row->employee_id) . '" class="flex items-center justify-center w-8 h-8 text-white bg-green-600 rounded-md edit"><i class="text-xl bx bx-show"></i></a>
                
                     <button onclick="openDeleteModalsched(' . $row->id . ')" class="flex items-center justify-center w-8 h-8 text-white bg-green-600 rounded-md edit">
                    <i class="text-xl bx bx-trash-alt"></i>
                </button>

                </div>';
                })
                ->addColumn('status', function ($row) {
                    $status = 0;
                    if ($row->status == null) {
                        $status = 'Pending';
                    } else {
                        $status = 'Approved';
                    }
                    return $status;
                })
                ->addColumn('time', function ($row) {
                    // Check if the shift is assigned and format the shift start and end times in 12-hour AM/PM format
                    if ($row->shift) {
                        $shiftStart = \Carbon\Carbon::parse($row->shift->shift_start)->format('g:i A');
                        $shiftEnd = \Carbon\Carbon::parse($row->shift->shift_end)->format('g:i A');
                        return "{$shiftStart} - {$shiftEnd}";
                    } else {
                        return 'No shift assigned';
                    }
                })
                ->addColumn('break', function ($row) {
                    // Check if the shift is assigned and format the break out and break in times in 12-hour AM/PM format
                    if ($row->shift) {
                        $breakOut = \Carbon\Carbon::parse($row->shift->break_out)->format('g:i A');
                        $breakIn = \Carbon\Carbon::parse($row->shift->break_in)->format('g:i A');
                        return "{$breakOut} - {$breakIn}";
                    } else {
                        return 'No shift assigned';
                    }
                })
                ->addColumn('employee_name', function ($row) {
                    return $row->employee ? $row->employee->first_name . ' ' . $row->employee->last_name : 'No Employee';
                })
                ->rawColumns(['action', 'approve'])
                ->make(true);
        }
    }

    public function approveSchedule(Request $request)
    {
        $schedule = WorkSchedule::find($request->id);
        if ($schedule) {
            $schedule->status = 'Approved';
            $schedule->save();
            return response()->json(['success' => 'Schedule approved successfully']);
        }
        return response()->json(['error' => 'Schedule not found'], 404);
    }



    public function employeesched(Request $request, $id)
    {
        $privileges = $request->get('privileges');
        if ($request->ajax()) {
            $data = WorkSchedule::with('shift', 'employee')->where('employee_id', $id)->get();

            if ($data->isEmpty()) {
                return response()->json([]);
            }

            return DataTables::of($data)
                ->addColumn('action', function ($row) use ($privileges) {
                    // If user HAS the "Employee UserInterface" privilege → restricted
                    if (in_array('Employee UserInterface', $privileges)) {
                        if ($row->status === null) {
                            return ' <div class="flex items-center justify-center gap-2">
            <button onclick="openDeleteModalsched(' . $row->id . ')" 
                    class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md">
                <i class="text-xl bx bx-trash-alt"></i>
            </button>
        </div>';
                            ;
                        } else {
                            return '<span class="text-gray-500 italic">Already Approved</span>';
                        }
                    }

                    // If user does NOT have the privilege → GOD access (can always delete)
                    return '
        <div class="flex items-center justify-center gap-2">
            <button onclick="openDeleteModalsched(' . $row->id . ')" 
                    class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md">
                <i class="text-xl bx bx-trash-alt"></i>
            </button>
        </div>';
                })


                ->addColumn('status123', function ($row) {

                    if ($row->status == null) {
                        $status = 'Pending';
                    } else {
                        $status = 'Approved';
                    }
                    return $status;
                })
                ->addColumn('time', function ($row) {
                    if ($row->shift) {
                        return date('h:i A', strtotime($row->shift->shift_start)) . ' to ' . date('h:i A', strtotime($row->shift->shift_end));
                    }
                    return 'No shift assigned';
                })
                ->addColumn('break', function ($row) {
                    if ($row->shift) {
                        return date('h:i A', strtotime($row->shift->break_out)) . ' to ' . date('h:i A', strtotime($row->shift->break_in));
                    }
                    return 'No break assigned';
                })


                ->rawColumns(['action'])
                ->make(true);
        }
    }



    // Store a new shift
    public function store(Request $request)
    {
        $request->validate([
            'shift_name' => 'required|string|max:100',
            'shift_start' => 'required',
            'shift_end' => 'required',
            'break_out' => 'nullable',
            'break_in' => 'nullable',
            'status' => 'required',
        ]);

        $shift = Shift::create($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Shift created successfully!',
            'data' => $shift
        ]);
    }


    public function store_sched(Request $request)
    {
        $validated = $request->validate([
            'employee_id' => 'required|integer|exists:employees,id',
            'shift_id' => 'required|integer|exists:shifts,id',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        $employeeId = $validated['employee_id'];
        $newStartDate = Carbon::parse($validated['start_date']);
        $newEndDate = isset($validated['end_date']) ? Carbon::parse($validated['end_date']) : $newStartDate;

        // Find overlapping schedules
        $existingSchedules = WorkSchedule::where('employee_id', $employeeId)
            ->where(function ($query) use ($newStartDate, $newEndDate) {
                $query->whereBetween('start_date', [$newStartDate, $newEndDate])
                    ->orWhereBetween('end_date', [$newStartDate, $newEndDate])
                    ->orWhere(function ($q) use ($newStartDate, $newEndDate) {
                        $q->where('start_date', '<', $newStartDate)
                            ->where('end_date', '>', $newEndDate);
                    });
            })
            ->orderBy('start_date')
            ->get();

        $overlapExists = false; // Track if the new schedule overrides an existing one

        foreach ($existingSchedules as $schedule) {
            $oldStartDate = Carbon::parse($schedule->start_date);
            $oldEndDate = Carbon::parse($schedule->end_date);

            if ($newStartDate > $oldStartDate && $newEndDate < $oldEndDate) {
                // If new schedule is in the middle, split into two
                WorkSchedule::create([
                    'employee_id' => $employeeId,
                    'shift_id' => $schedule->shift_id,
                    'start_date' => $oldStartDate,
                    'end_date' => $newStartDate->copy()->subDay(),
                    'status' => $schedule->status, // Keep original status
                ]);

                WorkSchedule::create([
                    'employee_id' => $employeeId,
                    'shift_id' => $schedule->shift_id,
                    'start_date' => $newEndDate->copy()->addDay(),
                    'end_date' => $oldEndDate,
                    'status' => $schedule->status, // Keep original status
                ]);

                $schedule->delete();
                $overlapExists = true;
            } elseif ($newStartDate <= $oldStartDate && $newEndDate >= $oldEndDate) {
                // If the new schedule fully overlaps an existing one, delete the old schedule
                $schedule->delete();
                $overlapExists = true;
            } elseif ($newStartDate <= $oldEndDate && $newStartDate > $oldStartDate) {
                // Adjust existing schedule's end date if new schedule starts in the middle
                $schedule->end_date = $newStartDate->copy()->subDay();
                $schedule->save();
                $overlapExists = true;
            } elseif ($newEndDate >= $oldStartDate && $newEndDate < $oldEndDate) {
                // Adjust existing schedule's start date if new schedule ends in the middle
                $schedule->start_date = $newEndDate->copy()->addDay();
                $schedule->save();
                $overlapExists = true;
            }
        }

        // Set status: Pending if no overrides, otherwise keep as pending (to be approved manually)
        $newScheduleStatus = $overlapExists ? null : null;

        // Insert the new schedule
        $shift = WorkSchedule::create([
            'employee_id' => $employeeId,
            'shift_id' => $validated['shift_id'],
            'start_date' => $newStartDate,
            'end_date' => $newEndDate,
            'status' => $newScheduleStatus, // Always Pending
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Shift created successfully (Pending approval), existing schedules adjusted if necessary!',
            'data' => $shift
        ], 201);
    }


    // Get a single shift
    public function show($id)
    {
        return response()->json(Shift::findOrFail($id));
    }

    // Update a shift
    public function update(Request $request, Shift $shift)
    {
        $request->validate([
            'shift_name' => 'required|string|max:255',
            'shift_start' => 'required',
            'shift_end' => 'required',
            'break_out' => 'nullable',
            'break_in' => 'nullable',
            'status' => 'nullable|in:active,inactive',
        ]);

        $shift->update($request->all());

        return response()->json(['status' => 'success']);
    }

    // Delete a shift
    public function destroy($id)
    {
        Shift::findOrFail($id)->delete();

        return response()->json(['success' => true, 'message' => 'deleted successfully.']);
    }
    public function destroysched($id)
    {
        WorkSchedule::findOrFail($id)->delete();

        return response()->json(['success' => true, 'message' => 'deleted successfully.']);
    }
    public function getShiftStatus($id)
    {
        $employee = Employee::findOrFail($id);
        return response()->json(['is_shift' => $employee->is_shift]);
    }
    public function updateShiftStatus(Request $request)
    {
        $employee = Employee::findOrFail($request->employee_id);
        $employee->is_shift = $request->is_shift;
        $employee->save();

        return response()->json(['message' => 'Shift status updated successfully!']);
    }
}
