<?php

namespace App\Http\Controllers;

use App\Models\Assign;
use App\Models\Employee;
use Illuminate\Http\Request;
use App\Models\AssignCompany;
use App\Models\OvertimeRequest;
use Illuminate\Support\Facades\Auth;
use App\Models\AssignCompanyDepartment;
use Illuminate\Database\QueryException;
use Yajra\DataTables\Facades\DataTables;
use Yajra\DataTables\Exceptions\Exception;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Models\Leave;

class OvertimeRequestController extends Controller
{
    public function upload(Request $request)
    {
        $image = $request->input('image');
        $image = str_replace('data:image/png;base64,', '', $image);
        $image = str_replace(' ', '+', $image);
        $imageName = 'face_' . time() . '.png';
        \File::put(public_path($imageName), base64_decode($image));

        return response()->json(['message' => 'Image saved']);
    }


    public function index(Request $request)
    {
        $user = Auth::user();

        // Get the agency IDs the user is assigned to
        $agencyIds = $user->viewable_agency_ids;



        $requests = OvertimeRequest::with(['employee', 'approver', 'department'])
            ->orderBy('created_at', 'desc');

        if (!empty($agencyIds)) {
            $requests->whereHas('department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', $restrictedCompanyIds)
                ->pluck('department_id')
                ->unique();

            if ($restrictedDepartmentIds->isNotEmpty()) {
                $requests->whereIn('department_id', $restrictedDepartmentIds);
            }
        }


        if ($request->ajax()) {
            $data = $requests->get();
            return DataTables::of($data)
                ->addColumn('employee_name', function ($row) {
                    return ($row->employee->first_name ?? '') . ' ' . ($row->employee->middle_name ?? '') . ' ' . $row->employee->last_name;
                })
                ->addColumn('department', function ($row) {
                    return $row->department->department_name ?? '';
                })
                ->addColumn('type', function ($row) {
                    return ucfirst($row->type);
                })
                ->addColumn('status', function ($row) {
                    return $row->is_approved === null
                        ? 'Pending'
                        : ($row->is_approved ? 'Approved' : 'Rejected');
                })

                ->addColumn('action', function ($row) {
                    $actionBtn = '<div class="flex items-center justify-center gap-2">';

                    $actionBtn .= ' <a href="' . route('over_under_time_request.edit', $row->id) . '" class="flex items-center justify-center w-10 h-10 text-white  bg-blue-600 rounded-md edit">
                                         <i class="text-xl bx bx-show"></i>
                </a>';
                    $actionBtn .= '<button onclick="openDeleteModal(' . $row->id . ')" class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md"><i class="text-xl bx bx-trash-alt"></i></button>';


                    $actionBtn .= '</div>';
                    return $actionBtn;
                })

                ->rawColumns(['action'])
                ->make(true);
        }




        $employee = Employee::with('channelclient')->orderBy('created_at', 'desc');

        if (!empty($agencyIds)) {
            $employee->whereHas('channelclient.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', $restrictedCompanyIds)
                ->pluck('department_id')
                ->unique();


            // Apply company restriction only if there are any
            if ($restrictedDepartmentIds->isNotEmpty()) {

                $employee->whereHas('channelclient.department', function ($q) use ($restrictedDepartmentIds) {
                    $q->whereIn('department_id', $restrictedDepartmentIds);
                });

            }
        }
        $employees = $employee->get();








        return view('Admin.Overtime.index', compact('employees'));
    }

    public function edit($id)
    {
        $request = OvertimeRequest::with(['employee', 'department'])->findOrFail($id);

        return view('Admin.Overtime.edit', compact('request'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'type' => 'required|in:overtime,undertime',
            'department_id' => 'required|exists:department,id',
            'position' => 'required|string',
            'request_date' => 'required|date',
            'regular_start' => 'nullable|date_format:H:i',
            'regular_end' => 'nullable|date_format:H:i',
            'actual_start' => 'nullable|date_format:H:i',
            'actual_end' => 'nullable|date_format:H:i|after:actual_start',
            'total_hours' => 'required|numeric|min:0.25',
            'reason' => 'nullable|string',
        ]);

        try {
            $request = OvertimeRequest::create($validated);

            return response()->json([
                'message' => 'Request submitted successfully.',
                'data' => $request,
            ], 201);
        } catch (QueryException $e) {
            // Catches DB errors like constraint violations
            return response()->json([
                'message' => 'Database error: ' . $e->getMessage(),
            ], 500);
        } catch (Exception $e) {
            // Generic catch-all for other unexpected errors
            return response()->json([
                'message' => 'Unexpected error: ' . $e->getMessage(),
            ], 500);
        }
    }


    public function update(Request $request, $id)
    {
        // Validate incoming request
        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'type' => 'required|in:overtime,undertime',
            'department_id' => 'required|exists:department,id',
            'position' => 'required|string',
            'request_date' => 'required|date',
            'regular_start' => 'nullable',
            'regular_end' => 'nullable',
            'actual_start' => 'nullable',
            'actual_end' => 'nullable|after:actual_start',
            'total_hours' => 'required|numeric|min:0.25',
            'reason' => 'nullable|string',
        ]);
        $overtimeRequest = OvertimeRequest::findOrFail($id);


        $overtimeRequest->update($validated);

        return redirect()
            ->route('over_under_time_request.index')
            ->with('success', 'Request updated successfully.');
    }



    public function destroy($id)
    {
        $overtime = OvertimeRequest::find($id);
        $overtime->delete();

        return response()->json(['message' => 'request record deleted successfully!']);
    }

    public function printRequest($id)
    {

        $request = OvertimeRequest::with(['employee', 'department'])->find($id);
        // Generate PDF
        $pdf = Pdf::loadView('Admin.Overtime.print', compact('request'));
        return $pdf->download('request.pdf');
    }
    public function updateStatus(Request $request, $id)
    {
        $overtimeRequest = OvertimeRequest::findOrFail($id);

        $overtimeRequest->is_approved = $request->input('is_approved');
        $overtimeRequest->approved_by = Auth::id();
        $overtimeRequest->approved_at = now();
        $overtimeRequest->save();

        $statusText = $request->input('is_approved') == 1 ? 'approved' : 'rejected';

        return redirect()->back()->with('success', "Request successfully {$statusText}.");
    }
    public function printleave($id)
    {

        $leave = leave::with(['employee'])->find($id);
        // Generate PDF
        $pdf = Pdf::loadView('Admin.Employee.leave.print', compact('leave'));
        return $pdf->download('leaverequest.pdf');
    }
}
