<?php



namespace App\Http\Controllers;

use App\Models\EmployeeClientChannel;
use App\Models\Leave;
use App\Models\Employee;

use App\Models\Assign;
use App\Models\AssignCompany;
use App\Models\AssignCompanyDepartment;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Facades\Auth;

class LeaveController extends Controller
{
    // Display a listing of the leaves
    public function index(Request $request)
    {
        $privileges = $request->get('privileges'); // Get the privileges from the middleware

        $user = Auth::user();

        // Get the agency IDs the user is assigned to
        $agencyIds = $user->viewable_agency_ids;

        $query = Leave::with('employee.channelclient');

        if (!empty($agencyIds)) {

            $query->whereHas('employee.channelclient.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id'))
                ->pluck('department_id')
                ->unique();

            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereHas('employee.channelclient', function ($q) use ($restrictedDepartmentIds) {
                    $q->whereIn('department_id', $restrictedDepartmentIds);
                });
            }
        }









        if ($request->ajax()) {
            $data = $query->orderBy('created_at', 'desc')->get();

            return DataTables::of($data)
                ->addColumn('employee', function ($row) {
                    return $row->employee->first_name . ' ' . strtoupper(substr($row->employee->middle_name, 0, 1)) . '. ' . $row->employee->last_name;
                })
                ->addColumn('paid', function ($row) {
                    return '
                        <div class="flex items-center justify-center gap-2">
                            <input type="checkbox" ' . ($row->paid ? 'checked' : '') . '  class="form-checkbox bg-blue-600 h-5 w-5 text-blue-600"> Paid
                        </div>
                    ';
                })
                ->addColumn('action', function ($row) use ($privileges) {
                    $actionBtn = '<div class="flex items-center justify-center gap-2">';

                    if (in_array('Manage Agency', $privileges)) {
                        $actionBtn .= '<a href="' . url('/admin/pre-profile/' . $row->employee_id) . '" class="flex items-center justify-center w-10 h-10 text-white bg-green-600 rounded-md edit"><i class="text-xl bx bx-show"></i></a>';
                    }

                    if (in_array('Edit Agency', $privileges)) {
                        $actionBtn .= '<a href="' . route('leaves.edit', $row->id) . '" class="flex items-center justify-center w-10 h-10 text-white bg-blue-600 rounded-md"><i class="bx bx-edit-alt"></i></a>';
                    }

                    if (in_array('Delete Agency', $privileges)) {
                        $actionBtn .= '<button onclick="openDeleteModal(' . $row->id . ')" class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md"><i class="text-xl bx bx-trash-alt"></i></button>';
                    }

                    $actionBtn .= '</div>';
                    return $actionBtn;
                })
                ->rawColumns(['paid', 'action'])
                ->make(true);
        }

        $data = leave::with('employee')->orderBy('created_at', 'desc')->get();


        $query = EmployeeClientChannel::with('employee.user', 'employee', 'department.client.agency')
            ->where('status', 1);
        if (!empty($agencyIds)) {

            $query->whereHas('department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            // Get the company IDs restricted via assignCompanies
            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', $restrictedCompanyIds)
                ->pluck('department_id')
                ->unique();


            // Apply company restriction only if there are any
            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereIn('department_id', $restrictedDepartmentIds);
            }
        }


        $employees = $query->get();
        return view('Admin.Employee.leave.index', compact('data', 'employees'));
    }



    public function approve($id)
    {
        $leave = Leave::findOrFail($id);
        $leave->status = 1; // Approved
        $leave->save();

        return response()->json(['message' => 'Leave approved successfully.']);
    }


    public function employeeleave(Request $request, $id)
    {
        $privileges = $request->get('privileges'); // Get the privileges from the middleware

        if ($request->ajax()) {
            $data = Leave::with('employee')->orderBy('created_at', 'desc')->where('employee_id', $id)->get();

            return DataTables::of($data)
                ->addColumn('employee', function ($row) {
                    return $row->employee->first_name . ' ' . strtoupper(substr($row->employee->middle_name, 0, 1)) . '. ' . $row->employee->last_name;
                })
                ->addColumn('paid', function ($row) {
                    return '
                        <div class="flex items-center justify-center gap-2">
                            <input type="checkbox" ' . ($row->paid ? 'checked' : '') . '  class="form-checkbox bg-blue-600 h-5 w-5 text-blue-600"> Paid
                        </div>
                    ';
                })
                ->addColumn('action', function ($row) use ($privileges) {
                    $actionBtn = '<div class="flex items-center justify-center gap-2">';



                    if (in_array('Delete Agency', $privileges)) {
                        $actionBtn .= '<button onclick="openDeleteModal(' . $row->id . ')" class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md"><i class="text-xl bx bx-trash-alt"></i></button>';
                    }

                    $actionBtn .= '</div>';
                    return $actionBtn;
                })
                ->rawColumns(['paid', 'action'])
                ->make(true);
        }
    }

    public function create()
    {
        return view('leaves.create');
    }

    // Store a newly created leave in storage
    public function store(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'leave_type' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'days' => 'required|integer|min:1',
            'paid' => 'required|boolean',
            'reason' => 'nullable|string',
        ]);

        $leave = Leave::create($request->all());

        return response()->json([
            'message' => "Leave request for {$leave->days} days has been saved successfully.",
        ], 200);
    }

    // Display the specified leave
    public function show(Leave $leave)
    {
        return view('leaves.show', compact('leave'));
    }

    // Show the form for editing the specified leave
    public function edit($id)
    {
        $leave = Leave::find($id);
        $employees = Employee::get();
        return view('Admin.Employee.leave.edit', compact('leave', 'employees'));
    }

    // Update the specified leave in storage
    public function update(Request $request, Leave $leave) // Should be $leave, NOT $leaf
    {
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'leave_type' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'days' => 'required|integer|min:1',
            'paid' => 'required|boolean',
            'reason' => 'nullable|string',
        ]);

        $leave->update($request->all());

        return redirect()->route('leaves.index')->with('success', 'Leave updated successfully.');
    }



    // Remove the specified leave from storage
    public function destroy($id)
    {
        $leave = Leave::find($id);
        $leave->delete();

        return response()->json(['message' => 'Leave record deleted successfully!']);
    }
}
