<?php

namespace App\Http\Controllers;

use App\Models\Employeerate;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class EmployeerateController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $query = Employeerate::with('employee'); // Include the employee relationship

            return DataTables::of($query)
                ->addColumn('employee_name', function ($row) {
                    return $row->employee ? $row->employee->name : 'N/A';
                })
                ->editColumn('monthly_salary', function ($row) {
                    return number_format($row->monthly_salary, 2);
                })
                ->addColumn('actions', function ($row) {
                    return '<a href="' . route('employeerates.edit', $row->id) . '" class="btn btn-sm btn-primary">Edit</a>
                            <button class="btn btn-sm btn-danger delete-btn" data-id="' . $row->id . '">Delete</button>';
                })
                ->rawColumns(['actions']) // Allow HTML in the actions column
                ->make(true);
        }
    }

    public function employee_rates(Request $request, $id)
    {
        if ($request->ajax()) {
            $data = Employeerate::with('employee')->where('employee_id', $id)->get();
            return DataTables::of($data)
                ->addColumn('employee_name', function ($row) {
                    return $row->employee ? $row->employee->name : 'N/A';
                })
                ->editColumn('monthly_salary', function ($row) {
                    return number_format($row->monthly_salary, 2);
                })
                ->addColumn('action', function ($row) {
                    // Action buttons (edit, delete)
                    $actionBtn = '<div class="flex items-center justify-center gap-2">
                <button class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md delete-btn" 
                data-url="' . route('employeerates.destroy', $row->id) . '" data-id="' . $row->id . '">
                <i class="text-xl bx bx-trash-alt"></i>
                </button>
            </div>';
                    return $actionBtn;
                })
                ->make(true);
        }
    }



    public function updateStatus(Request $request, $id)
    {
        $rate = Employeerate::findOrFail($id);
        $rate->status = $request->status;
        $rate->save();

        return response()->json(['success' => true]);
    }


    public function create()
    {
        return view('employeerates.create'); // Create form view
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'status' => 'nullable|string',
            'date' => 'nullable|date',
            'monthly_salary' => 'nullable|numeric',
            'daily_rate' => 'nullable|numeric',
            'cola_1' => 'nullable|numeric',
            'cola_2' => 'nullable|numeric',
            'allowance' => 'nullable|numeric',
        ]);

        $employeerate = Employeerate::create($validated);

        return redirect()->back()->with('success', 'Employee rate created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $employeerate = Employeerate::with('employee')->findOrFail($id);

        return view('employeerates.show', compact('employeerate')); // Details page
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $employeerate = Employeerate::findOrFail($id);

        return view('employeerates.edit', compact('employeerate')); // Edit form view
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $employeerate = Employeerate::findOrFail($id);

        $validated = $request->validate([
            'status' => 'nullable|string',
            'date' => 'nullable|date',
            'monthly_salary' => 'nullable|numeric',
            'daily_rate' => 'nullable|numeric',
            'cola_1' => 'nullable|numeric',
            'cola_2' => 'nullable|numeric',
            'allowance' => 'nullable|numeric',
        ]);

        $employeerate->update($validated);

        return redirect()->route('employeerates.index')->with('success', 'Employee rate updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $employeerate = Employeerate::findOrFail($id);
        $employeerate->delete();

        return response()->json(['message' => 'Employee rate deleted successfully!']);
    }
}
