<?php

namespace App\Http\Controllers;

use App\Models\DepartmentStore;

use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use App\Models\User;
use App\Models\Client;
use App\Models\Department;
use App\Models\Employee;
use App\Models\Charges;
use App\Models\AssignCompany;
use App\Models\AssignCompanyDepartment;
use App\Models\Assign;
use App\Models\ChargeGroup;
use App\Models\TrxCode;
use App\Models\DepartmentFile;
use App\Models\EmployeeClientChannel;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Carbon\Carbon;
use App\Models\StoreDepartment;
use App\Models\Offices;
use App\Models\Stores;
use Illuminate\Support\Facades\Crypt;

class DepartmentController extends Controller
{


    public function forapporval(Request $request)
    {
        if ($request->ajax()) {
            $query = Employee::with('user');

            // Apply filters if present
            if ($request->filled('office_id')) {
                $query->where('office_id', $request->office_id);
            }

            if ($request->filled('store_id')) {
                $query->where('store_id', $request->store_id);
            }

            if ($request->has('pwd_filter')) {
                $pwdFilter = $request->pwd_filter;
                if ($pwdFilter === 'Not PWD') {
                    $query->where('pwd', 'Not PWD');
                } elseif ($pwdFilter === 'PWD') {
                    $query->where('pwd', 'PWD');
                }
            }

            // ✅ New filter: Has or doesn't have department
            if ($request->has('has_department')) {
                if ($request->has_department === 'yes') {
                    $query->whereHas('channelclient', function ($q) {
                        $q->whereNotNull('department_id');
                    });
                } elseif ($request->has_department === 'no') {
                    $query->where(function ($q) {
                        $q->whereDoesntHave('channelclient')
                            ->orWhereHas('channelclient', function ($sub) {
                                $sub->whereNull('department_id');
                            });
                    });
                }
            }

            $data = $query->get();

            return DataTables::of($data)
                ->addColumn('action', function ($row) {
                    return '<div class="flex items-center justify-center gap-2">
                    <a href="' . url('admin/pre-profile/' . Crypt::encrypt($row->id)) . '" class="flex items-center justify-center w-8 h-8 text-white bg-green-600 rounded-md edit"><i class="text-xl bx bx-show"></i></a>
                    <button onclick="openDeleteModal(' . $row->id . ')" class="flex items-center justify-center w-8 h-8 text-white bg-green-600 rounded-md">
                        <i class="text-xl bx bx-trash-alt"></i>
                    </button>
                </div>';
                })
                ->addColumn('mrf', function ($row) {
                    return '<div class="flex items-center justify-center gap-2">
                    <a href="' . url('admin/pre-profile/' . Crypt::encrypt($row->id)) . '" class="flex items-center justify-center w-8 h-8 text-white bg-green-600 rounded-md edit"><i class="text-xl bx bx-show"></i></a>
                </div>';
                })
                ->addColumn('status', function ($row) {
                    return $row->status == 1 ? "Walk-in Applicant" : "Online Applicant";
                })
                ->addColumn('profile', function ($row) {
                    $imageUrl = $row->profile ? asset($row->profile) : asset('/default/image.png');
                    return '<div class="flex items-center justify-center"><img src="' . $imageUrl . '" alt="Profile Image" style="width: 50px; height: 50px; border-radius: 50%;"></div>';
                })
                ->addColumn('name', function ($row) {
                    return '<div class="max-w-xs overflow-hidden whitespace-nowrap text-ellipsis">' .
                        htmlspecialchars($row->first_name . ' ' . $row->middle_name . ' ' . $row->last_name) .
                        '</div>';
                })
                ->rawColumns(['action', 'name', 'mrf', 'profile'])
                ->make(true);
        }
    }




    public function getAllStores($departmentId)
    {
        // get all store IDs linked to this department
        $storeIds = StoreDepartment::where('department_id', $departmentId)
            ->pluck('store_id')
            ->toArray();

        // fetch the stores
        $stores = Stores::whereIn('id', $storeIds)
            ->select('id', 'name')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $stores
        ]);
    }



    public function index_file(Request $request, $id)
    {
        if ($request->ajax()) {
            $data = DepartmentFile::where('department_id', $id)->get();

            return DataTables::of($data)
                ->addColumn('file_link', function ($row) {
                    if ($row->file) {
                        return '<a href="' . asset($row->file) . '" target="_blank" class="btn btn-primary">View File</a>';
                    }
                    return 'No File';
                })
                ->addColumn('action', function ($row) {
                    return ' <div class="flex items-center justify-center gap-2"><button    class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md delete-file" data-id="' . $row->id . '"> <i class="text-xl bx bx-trash-alt"></i></button> </div>';
                })




                ->rawColumns(['file_link', 'action'])
                ->make(true);
        }
    }
    public function assignedstore(Request $request, $id)
    {
        if ($request->ajax()) {
            $data = Stores::where('department_id', $id)->get();

            return DataTables::of($data)

                ->addColumn('action', function ($row) {
                    $editBtn = '<button class="flex items-center justify-center w-10 h-10 text-white bg-yellow-500 rounded-md edit-store" data-id="' . $row->id . '">
                                <i class="text-xl bx bx-edit-alt"></i>
                            </button>';

                    $deleteBtn = '<button class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md delete-file" data-id="' . $row->id . '">
                                <i class="text-xl bx bx-trash-alt"></i>
                            </button>';

                    return '<div class="flex items-center justify-center gap-2">' . $editBtn . $deleteBtn . '</div>';
                })

                ->rawColumns(['action'])
                ->make(true);
        }
    }


    public function storeDepartmentStore(Request $request)
    {
        $request->validate([
            'department_id' => 'required|exists:department,id',
            'store_id' => 'required|exists:stores,id',
        ]);

        // Prevent duplicate entry
        $exists = DepartmentStore::where('department_id', $request->department_id)
            ->where('store_id', $request->store_id)
            ->exists();

        if ($exists) {
            return response()->json([
                'success' => false,
                'message' => 'This store is already assigned to the department.'
            ], 409);
        }

        DepartmentStore::create([
            'department_id' => $request->department_id,
            'store_id' => $request->store_id,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Store assigned to department successfully.'
        ]);
    }
    public function deletestore($id)
    {

        $store = Stores::find($id);
        $store->delete();
        $user = User::find($store->user_id);
        $user->delete();
        return response()->json($store, 200);
    }

    public function delete_file($id)
    {
        $file = DepartmentFile::find($id);

        if (!$file) {
            return response()->json(['error' => 'File not found!'], 404);
        }

        // Correct file path
        $filePath = public_path($file->file);

        // Check if the file exists before deleting it
        if (!empty($file->file) && file_exists($filePath)) {
            unlink($filePath);
        }

        // Delete the record from the database
        $file->delete();

        return response()->json(['message' => 'File deleted successfully!']);
    }


    public function store_file(Request $request)
    {
        // Validate the request
        $request->validate([
            'department_id' => 'required|exists:department,id',
            'name' => 'required|string|max:255',
            'file' => 'required|file|max:51200', // Max file size: 50MB
        ]);

        $filePath = null;

        // Handle file upload if provided
        if ($request->hasFile('file')) {
            $file = $request->file('file');
            $fileName = time() . '_' . $file->getClientOriginalName();
            $filePath = 'uploads/' . $fileName;
            $file->move(public_path('uploads'), $fileName); // Move file to public/uploads
        }

        // Create a new record in the database
        $fileEntry = DepartmentFile::create([
            'department_id' => $request->department_id,
            'name' => $request->name,
            'file' => $filePath, // Store the relative path
        ]);

        return response()->json([
            'message' => 'File uploaded successfully!',
            'file' => $fileEntry
        ]);
    }

    public function updateStatus(Request $request)
    {
        // Validate request
        $request->validate([
            'id' => 'required|exists:assignemployeee,id', // Use the correct table name
            'status' => 'nullable|string',
            'cutomsetup' => 'nullable',
            'sss' => 'nullable',
            'pagibig' => 'nullable',
            'philhealth' => 'nullable'
        ]);


        // Find the history record
        $history = EmployeeClientChannel::find($request->id);

        // Check if the record exists before updating
        if (!$history) {
            return response()->json([
                'message' => 'Record not found'
            ], 404);
        }

        if ($request->has('status')) {
            $history->status = $request->status;
        }
        if ($request->has('cutomsetup')) {
            $history->cutomsetup = $request->cutomsetup;
        }
        if ($request->has('sss')) {
            $history->sss = $request->sss;
        }
        if ($request->has('pagibig')) {
            $history->pagibig = $request->pagibig;
        }
        if ($request->has('philhealth')) {
            $history->philhealth = $request->philhealth;
        }
        // Update status

        $history->save();

        // Return proper JSON response
        return response()->json([
            'message' => 'Status updated successfully',
            'status' => $history->status
        ], 200);
    }



    public function depratmentemployees(Request $request, $id)
    {
        $department = Department::find($id);
        $trx = TrxCode::get();


        if ($request->ajax()) {
            $data = EmployeeClientChannel::with('department.client.agency', 'employee')->where('department_id', $id)->get();

            return DataTables::of($data)
                ->addColumn('status', function ($row) {

                    if ($row->status == 1) {
                        $status = 'Active';
                    } else {
                        $status = 'Inactive';
                    }
                    return $status;
                })
                ->addColumn('employee_name', function ($row) {
                    return $row->employee->first_name . ' ' . ($row->employee->middle_name ?? '') . ' ' . $row->employee->last_name;
                })

                ->addColumn('action', function ($row) {
                    return '<div class="flex items-center justify-center gap-2">

                            <a href="' . route('pre.profile', Crypt::encrypt($row->employee_id)) . '" class="flex items-center justify-center w-10 h-10 text-white bg-green-600 rounded-md edit">
                                <i class="text-xl bx bx-show"></i>
                            </a>
                            <button onclick="openClientEmployeeDeleteModal(' . $row->id . ')" class="flex items-center justify-center  w-10 h-10 text-white bg-green-600 rounded-md">
                        <i class="text-xl bx bx-trash-alt"></i>
                    </button>
                        </div>';
                })
                ->make(true);
        }
        return view('Admin.Department.employees', compact(
            'department',
            'trx'
        ));
    }

    public function updateEmployeeField(Request $request)
    {
        $request->validate([
            'id' => 'required|exists:assignemployeee,id',
            'field' => 'required|string|in:date_hired,position,employment_start_date,dtr_id,employment_end_date,entry_time,exit_time,store_id',
            'value' => 'required'
        ]);

        $employee = EmployeeClientChannel::findOrFail($request->id);
        $employee->{$request->field} = $request->value;
        $employee->save();

        return response()->json(['success' => true, 'message' => 'Field updated successfully.']);
    }



    public function charges(Request $request, $id)
    {
        if ($request->ajax()) {
            $now = Carbon::now();

            $charges = Charges::where('department_id', $id)->get();


            return DataTables::of($charges)
                ->addColumn('status', function ($row) {

                    $checked = $row->status == 1 ? 'checked' : '';
                    return '<input type="checkbox" class="status-toggle" data-id="' . $row->id . '" ' . $checked . '>';
                })
                ->addColumn('action', function ($row) {
                    return '<div class="flex items-center justify-center gap-2">
                                <a href="' . route('departments.editCharges', $row->id) . '" class="flex items-center justify-center w-10 h-10 text-white bg-blue-600 rounded-md"><i class="bx bx-edit-alt"></i></a>
                
                                <button class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md delete-btn" 
                                        data-id="' . $row->id . '">
                                    <i class="text-xl bx bx-trash"></i>
                                </button>
                            </div>';
                })

                ->addColumn('date', function ($row) {
                    return Carbon::parse($row->start_date)->format('F j, Y') . ' - ' . Carbon::parse($row->end_date)->format('F j, Y');
                })
                ->rawColumns(['status', 'action'])
                ->make(true);
        }
    }





    public function editCharges($id)
    {
        // Fetch the charges by its ID
        $charges = Charges::with('department')->findOrFail($id);

        // Get all service types and VAT types to populate the dropdowns
        $allServiceTypes = TrxCode::all();
        $allVatTypes = TrxCode::all();

        $allWithTypes = TrxCode::all();
        // Get the current service and VAT types linked to the charges
        $serviceTypes = ChargeGroup::where('charges_id', $charges->id)
            ->where('type', 'service')
            ->pluck('group_code_id')
            ->toArray();

        $vatTypes = ChargeGroup::where('charges_id', $charges->id)
            ->where('type', 'vat')
            ->pluck('group_code_id')
            ->toArray();

        $WithTypes = ChargeGroup::where('charges_id', $charges->id)
            ->where('type', 'with')
            ->pluck('group_code_id')
            ->toArray();


        // Pass all data to the view
        return view('Admin.Department.settings.charges_modal', compact('WithTypes', 'charges', 'serviceTypes', 'vatTypes', 'allServiceTypes', 'allWithTypes', 'allVatTypes'));
    }

    public function updateCharges(Request $request, $id)
    {
        // Validate incoming data
        $data = $request->validate([
            'service_charge' => 'nullable|numeric',
            'charge_amount' => 'nullable|numeric',
            'special_rate_service' => 'nullable|numeric',
            'special_rate_vat' => 'nullable|numeric',
            'special_rate_with' => 'nullable|numeric',
            'vat' => 'required|numeric',
            'withholding_tax' => 'required|numeric',
            'service_charge_type' => 'required|string|max:255',
            'convert_inminute' => 'nullable|string|max:255',
            'service_type' => 'nullable|array',
            'vat_type' => 'nullable|array',
            'with_type' => 'nullable|array',
            's_camputation_for_all' => 'nullable|numeric',
            's_sss' => 'nullable|numeric',
            's_phil' => 'nullable|numeric',
            's_pagibig' => 'nullable|numeric',
            'v_camputation_for_all' => 'nullable|numeric',
            'v_sss' => 'nullable|numeric',
            'v_phil' => 'nullable|numeric',
            'v_pagibig' => 'nullable|numeric',
            'w_camputation_for_all' => 'nullable|numeric',
            'w_sss' => 'nullable|numeric',
            'w_phil' => 'nullable|numeric',
            'w_pagibig' => 'nullable|numeric',
        ]);

        // Update the Charges
        $charges = Charges::findOrFail($id);
        $charges->update([
            'service_charge' => $data['service_charge'],
            'charge_amount' => $data['charge_amount'],
            'special_rate_service' => $data['special_rate_service'],
            'special_rate_vat' => $data['special_rate_vat'],
            'vat' => $data['vat'],
            'withholding_tax' => $data['withholding_tax'],
            'service_charge_type' => $data['service_charge_type'],
            'convert_inminute' => $data['convert_inminute'] ?? 0,
            'v_camputation_for_all' => $data['v_camputation_for_all'] ?? 0,
            'v_sss' => $data['v_sss'] ?? 0,
            'v_phil' => $data['v_phil'] ?? 0,
            'v_pagibig' => $data['v_pagibig'] ?? 0,
            'w_camputation_for_all' => $data['w_camputation_for_all'] ?? 0,
            'w_sss' => $data['w_sss'] ?? 0,
            'w_phil' => $data['w_phil'] ?? 0,
            'w_pagibig' => $data['w_pagibig'] ?? 0,
            's_camputation_for_all' => $data['s_camputation_for_all'] ?? 0,
            's_sss' => $data['s_sss'] ?? 0,
            's_phil' => $data['s_phil'] ?? 0,
            's_pagibig' => $data['s_pagibig'] ?? 0,
        ]);

        // Update the service and vat types
        ChargeGroup::where('charges_id', $charges->id)->where('type', 'service')->delete();
        ChargeGroup::where('charges_id', $charges->id)->where('type', 'vat')->delete();
        ChargeGroup::where('charges_id', $charges->id)->where('type', 'with')->delete();

        // Add the new service types
        if (!empty($data['service_type'])) {
            foreach ($data['service_type'] as $serviceTypeId) {
                ChargeGroup::create([
                    'charges_id' => $charges->id,
                    'group_code_id' => $serviceTypeId,
                    'type' => 'service',
                ]);
            }
        }

        // Add the new VAT types
        if (!empty($data['vat_type'])) {
            foreach ($data['vat_type'] as $vatTypeId) {
                ChargeGroup::create([
                    'charges_id' => $charges->id,
                    'group_code_id' => $vatTypeId,
                    'type' => 'vat',
                ]);
            }
        }
        if (!empty($data['with_type'])) {
            foreach ($data['with_type'] as $withTypeId) {
                ChargeGroup::create([
                    'charges_id' => $charges->id,
                    'group_code_id' => $withTypeId,
                    'type' => 'with',
                ]);
            }
        }

        // Redirect back with success message
        return redirect()->route('departments.employees.show', ['id' => $charges->department_id])
            ->with('success', 'Charges updated successfully.');
    }









    public function updateBillingStatus(Request $request, $id)
    {
        $department = Department::find($id);

        if (!$department) {
            return response()->json(['success' => false, 'message' => 'Department not found'], 404);
        }

        $department->billing_status = $request->billing_status;
        $department->save();

        return response()->json(['success' => true, 'message' => 'Billing status updated successfully']);
    }

    public function updatechargeStatus(Request $request)
    {
        $charge = Charges::find($request->id);
        if ($charge) {
            $charge->status = $request->status;
            $charge->save();
            return response()->json(['success' => true, 'message' => 'Status updated successfully.']);
        }
        return response()->json(['success' => false, 'message' => 'Charge not found.']);
    }






    public function delete($id)
    {
        $charge = Charges::find($id);
        if (!$charge) {
            return response()->json(['success' => false, 'message' => 'Charge not found'], 404);
        }

        $charge->delete();
        return response()->json(['success' => true, 'message' => 'Charge deleted successfully']);
    }

    public function deleteemployeedepartment($id)
    {
        $charge = EmployeeClientChannel::find($id);
        if (!$charge) {
            return response()->json(['success' => false, 'message' => 'Charge not found'], 404);
        }

        $charge->delete();
        return response()->json(['success' => true, 'message' => 'Charge deleted successfully']);
    }









    public function depratmentemployees1(Request $request)
    {
        $userId = auth()->id();



        $department = Department::where('user_id', $userId)->first();
        if ($request->ajax()) {
            $data = EmployeeClientChannel::with('department.client.agency', 'employee')->where('department_id', $department->id)->get();

            return DataTables::of($data)
                ->addColumn('status', function ($row) {

                    if ($row->status == 1) {
                        $status = 'Active';
                    } else {
                        $status = 'Inactive';
                    }
                    return $status;
                })
                ->addColumn('employment_start_date', function ($row) {
                    return \Carbon\Carbon::parse($row->employment_start_date)->format('F j, Y'); // Example: January 1, 2024
                })
                ->addColumn('employment_end_date', function ($row) {
                    return \Carbon\Carbon::parse($row->employment_end_date)->format('F j, Y'); // Example: January 1, 2024
                })

                ->addColumn('action', function ($row) {
                    return '<div class="flex items-center justify-center gap-2">

                            <a href="' . route('pre.profile', Crypt::encrypt($row->employee_id)) . '" class="flex items-center justify-center w-10 h-10 text-white bg-green-600 rounded-md edit">
                                <i class="text-xl bx bx-show"></i>
                            </a>
                        </div>';
                })
                ->make(true);
        }
        $trx = TrxCode::get();
        return view('Admin.Department.employees', compact('department', 'trx'));
    }




    public function index(Request $request)
    {
        $clients = Client::with('agency')->get();
        $assignedEmployees = EmployeeClientChannel::getAssignedEmployees();
        $notAssignedEmployees = EmployeeClientChannel::getNotAssignedEmployees();
        $trx = TrxCode::get();


        $user = Auth::user();

        // Get the agency IDs the user is assigned to
        $agencyIds = $user->viewable_agency_ids;


        $query = Department::with('client.agency')->withCount([
            'channels as active_employee_count' => function ($query) {
                $query->where('status', 1);
            }
        ]);

        if (!empty($agencyIds)) {

            $query->whereHas('client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            // Get the company IDs restricted via assignCompanies
            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', $restrictedCompanyIds)
                ->pluck('department_id')
                ->unique();


            // Apply company restriction only if there are any
            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereIn('id', $restrictedDepartmentIds);
            }
        }


        if ($request->ajax()) {

            $data = $query;

            return DataTables::of($data)
                ->addColumn('employee_count', function ($row) {
                    return $row->active_employee_count;
                })

                ->addColumn('action', function ($row) {
                    $departmentName = e($row->department_name); // escape for HTML
                    $assignUrl = route('departments.assignEmployee', $row->id);


                    return '<div class="flex items-center justify-center gap-2">
                    <a href="' . route('departments.edit', $row->id) . '" class="flex items-center justify-center w-10 h-10 text-white bg-blue-600 rounded-md edit">
                                           <i class="bx bx-edit-alt"></i>
                </a>
    
                     <button onclick="openAddModal(\'' . addslashes($departmentName) . '\', ' . $row->id . ', \'' . $assignUrl . '\')" 
            class="flex items-center justify-center w-10 h-10 text-white bg-green-600 rounded-md add-employee-btn">
            <i class="bx bxs-user-plus"></i>
        </button>
                                 <a href="' . route('departments.employees.show', $row->id) . '" class="flex items-center justify-center w-10 h-10 text-white bg-green-600 rounded-md edit">
                    <i class="text-xl bx bx-show"></i>
                </a>

               <button onclick="confirmDelete(' . $row->id . ')" class="px-2 py-1 w-10 h-10 text-white bg-red-500 rounded">
                                            <i class="text-sm bx bx-trash"></i>
                                          </button>
                </div>';
                })
                ->addColumn('entry_time', function ($row) {
                    return \Carbon\Carbon::parse($row->entry_time)->format('Y-m-d H:i:s');
                })
                ->addColumn('exit_time', function ($row) {
                    return \Carbon\Carbon::parse($row->exit_time)->format('Y-m-d H:i:s');
                })
                ->make(true);
        }

        $employees = Employee::with('user')->get();
        $offices = Offices::all();
        $stores = Stores::all();



        return view('Admin.Department.index', compact('clients', 'stores', 'offices', 'trx', 'notAssignedEmployees', 'assignedEmployees'));
    }
    public function fetchDepartments(Request $request, $id)
    {
        $user = Auth::user();

        $agencyIds = $user->viewable_agency_ids;

        $query = Department::with('client.agency');

        if (!empty($agencyIds)) {

            $query->whereHas('client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id'))
                ->pluck('department_id')
                ->unique();

            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereIn('id', $restrictedDepartmentIds);

            }
        }



        if ($request->ajax()) {
            $data = $query->where('client_id', $id)->withCount([
                'channels as active_employee_count' => function ($query) {
                    $query->where('status', 1);
                }
            ])->get();

            return DataTables::of($data)
                ->addColumn('employee_count', function ($row) {
                    return $row->active_employee_count;
                })

                ->addColumn('action', function ($row) {
                    return '<div class="flex items-center justify-center gap-2">
                  
    
                  
                                 <a href="' . route('departments.employees.show', $row->id) . '" class="flex items-center justify-center w-10 h-10 text-white bg-green-600 rounded-md edit">
                    <i class="text-xl bx bx-show"></i>
                </a>

                   
                </div>';
                })

                ->make(true);
        }
    }


    public function create()
    {
        return view('departments.create');
    }

    public function store(Request $request)
    {
        // Validate the incoming request data
        $data = $request->validate([
            'client_id' => 'required|exists:clients,id',
            'email' => 'required|string|email|max:255|unique:users',
            'start' => 'nullable|string',
            'end_date' => 'nullable|string',
            'department_name' => 'nullable|string',
            'service_charge' => 'nullable|numeric',
            'charge_amount' => 'nullable|numeric',
            'special_rate_vat' => 'nullable|numeric',
            'special_rate_service' => 'nullable|numeric',
            'vat' => 'required|numeric',
            's_camputation_for_all' => 'nullable|string',
            'v_camputation_for_all' => 'nullable|string',
            'withholding_tax' => 'required|numeric',
            'service_charge_type' => 'nullable|string|max:255',
            'convert_inminute' => 'nullable|string|max:255',
            'service_type' => 'nullable|array', // Ensure service_type is an array
            'service_type.*' => 'exists:trx_codes,id', // Validate each selected ID
            'vat_type' => 'nullable|array', // Ensure service_type is an array
            'vat_type.*' => 'exists:trx_codes,id', // Validate each selected ID


        ]);

        // Check if the department name already exists
        $existingDepartment = Department::where('department_name', $data['department_name'])->first();

        if ($existingDepartment) {
            return redirect()->back()->withErrors(['department_name' => 'The department name already exists.']);
        }

        // Create a User for the department
        $user = User::create([
            'name' => $data['department_name'], // Use the department name as the user's name
            'email' => $data['email'],
            'password' => Hash::make('123'),
            'role_id' => 8,
        ]);

        // Create the Department
        $department = Department::create([
            'user_id' => $user->id,
            'client_id' => $data['client_id'],
            'department_name' => $data['department_name'],
            'start' => $data['start'],
            'end' => $data['end_date'],
        ]);

        // Create Charges for the department
        $charges = Charges::create([
            'department_id' => $department->id,
            'service_charge' => $data['service_charge'],
            'charge_amount' => $data['charge_amount'],
            'vat' => $data['vat'],
            'withholding_tax' => $data['withholding_tax'],
            'service_charge_type' => $data['service_charge_type'],
            'convert_inminute' => $data['convert_inminute'] ?? null,
            'special_rate_service' => $data['special_rate_service'],
            'special_rate_vat' => $data['special_rate_vat'],
            's_camputation_for_all' => $data['s_camputation_for_all'] ?? 0, // Handle unchecked checkbox
            'v_camputation_for_all' => $data['v_camputation_for_all'] ?? 0,



        ]);

        if (!empty($data['service_type'])) {
            foreach ($data['service_type'] as $serviceTypeId) {
                ChargeGroup::create([
                    'charges_id' => $charges->id, // Assuming charges_id is linked to department_id
                    'group_code_id' => $serviceTypeId, // Assigning selected service type ID
                    'type' => 'service', // Set type as "service"
                ]);
            }
        }
        if (!empty($data['vat_type'])) {
            foreach ($data['vat_type'] as $vatTypeId) {
                ChargeGroup::create([
                    'charges_id' => $charges->id, // Assuming charges_id is linked to department_id
                    'group_code_id' => $vatTypeId, // Assigning selected service type ID
                    'type' => 'vat', // Set type as "service"
                ]);
            }
        }

        // Redirect with a success message
        return redirect()->back()->with('success', "Department '{$department->department_name}' created successfully and charges assigned.");
    }
    public function storecharges(Request $request)
    {
        try {
            // Validate incoming data
            $data = $request->validate([
                'department_id' => 'required|exists:department,id',
                'service_charge' => 'nullable|numeric',
                'charge_amount' => 'nullable|numeric',
                'special_rate_vat' => 'nullable|numeric',
                'special_rate_service' => 'nullable|numeric',
                'vat' => 'required|numeric',
                'withholding_tax' => 'required|numeric',
                'service_charge_type' => 'required|string|max:255',
                'convert_inminute' => 'required|string|max:255',
                'service_type' => 'nullable|array',
                'service_type.*' => 'exists:trx_codes,id',
                'vat_type' => 'nullable|array',
                'vat_type.*' => 'exists:trx_codes,id',

                'start_date' => 'nullable|string',
                'end_date' => 'nullable|string',
                's_camputation_for_all' => 'nullable|numeric',
                's_sss' => 'nullable|numeric',
                's_phil' => 'nullable|numeric',
                's_pagibig' => 'nullable|numeric',
                'v_camputation_for_all' => 'nullable|numeric',
                'v_sss' => 'nullable|numeric',
                'v_phil' => 'nullable|numeric',
                'v_pagibig' => 'nullable|numeric',
                'w_camputation_for_all' => 'nullable|numeric',
                'w_sss' => 'nullable|numeric',
                'w_phil' => 'nullable|numeric',
                'w_pagibig' => 'nullable|numeric',
            ]);

            // Create the charges record
            $charges = Charges::create([
                'department_id' => $data['department_id'],
                'service_charge' => $data['service_charge'] ?? null,
                'charge_amount' => $data['charge_amount'] ?? null,
                'vat' => $data['vat'],
                'withholding_tax' => $data['withholding_tax'],
                'service_charge_type' => $data['service_charge_type'],
                'convert_inminute' => $data['convert_inminute'],
                'special_rate_service' => $data['special_rate_service'] ?? null,
                'special_rate_vat' => $data['special_rate_vat'] ?? null,

                'start_date' => $data['start_date'],
                'end_date' => $data['end_date'],
                'v_camputation_for_all' => $data['v_camputation_for_all'] ?? 0,
                'v_sss' => $data['v_sss'] ?? 0,
                'v_phil' => $data['v_phil'] ?? 0,
                'v_pagibig' => $data['v_pagibig'] ?? 0,
                'w_camputation_for_all' => $data['w_camputation_for_all'] ?? 0,
                'w_sss' => $data['w_sss'] ?? 0,
                'w_phil' => $data['w_phil'] ?? 0,
                'w_pagibig' => $data['w_pagibig'] ?? 0,
                's_camputation_for_all' => $data['s_camputation_for_all'] ?? 0,
                's_sss' => $data['s_sss'] ?? 0,
                's_phil' => $data['s_phil'] ?? 0,
                's_pagibig' => $data['s_pagibig'] ?? 0,
            ]);

            // Assign service types if selected
            if (!empty($data['service_type'])) {
                foreach ($data['service_type'] as $serviceTypeId) {
                    ChargeGroup::create([
                        'charges_id' => $charges->id,
                        'group_code_id' => $serviceTypeId,
                        'type' => 'service',
                    ]);
                }
            }

            // Assign VAT types if selected
            if (!empty($data['vat_type'])) {
                foreach ($data['vat_type'] as $vatTypeId) {
                    ChargeGroup::create([
                        'charges_id' => $charges->id,
                        'group_code_id' => $vatTypeId,
                        'type' => 'vat',
                    ]);
                }
            }
            if (!empty($data['withholding_type'])) {
                foreach ($data['withholding_type'] as $vatTypeId) {
                    ChargeGroup::create([
                        'charges_id' => $charges->id,
                        'group_code_id' => $vatTypeId,
                        'type' => 'with',
                    ]);
                }
            }
            // Return success response in JSON format
            return response()->json([
                'success' => true,
                'message' => 'Charges have been successfully assigned.',
                'data' => $charges
            ], 200);
        } catch (\Exception $e) {
            // Return error response in JSON format
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while processing the request.',
                'error' => $e->getMessage()
            ], 500);
        }
    }



    public function show(Department $department)
    {

        $company = Client::all();


        return view('departments.show', compact('department', 'company', 'charges'));
    }

    public function edit(Department $department)
    {

        $company = Client::all();
        $charges = Charges::where('department_id', $department->id)->first();
        return view('Admin.Department.edit', compact('department', 'company', 'charges'));
    }

    public function update(Request $request, $id)
    {
        // Validate the request
        $data = $request->validate([
            'client_id' => 'required|exists:clients,id',
            'department_name' => 'required|string|max:255',
        ]);

        // Find and update the department
        $department = Department::findOrFail($id);
        $department->update([
            'client_id' => $data['client_id'],
            'department_name' => $data['department_name'],
        ]);

        // Redirect with success message
        return redirect()->route('departments.index')->with('success', "Department '{$department->department_name}' updated successfully.");
    }




    public function destroy(Department $department)
    {
        $department->delete();  // Delete department
        return response()->json(['success' => true, 'message' => 'Record deleted successfully.']);
    }
    public function getDepartments($clientId)
    {
        // Fetch departments for the selected client
        $departments = Department::where('client_id', $clientId)->get();

        // Return the departments as JSON
        return response()->json($departments);
    }



    public function assignEmployee(Request $request)
    {
        try {
            $request->validate([
                'department_id' => 'required|integer',
                'storeID' => 'nullable',
                'employee_ids' => 'required|array',
                'employee_ids.*' => 'integer',
                'position' => 'required|string|max:255',
                'date_hired' => 'required|date',
                'employment_start_date' => 'required|date',
                'employment_end_date' => 'nullable|date',
                'salary' => 'nullable|numeric',
                'entry_time' => 'nullable|string',
                'default_rate_id' => 'nullable',
                'exit_time' => 'nullable|string',
            ]);

            foreach ($request->employee_ids as $employeeId) {
                EmployeeClientChannel::where('employee_id', $employeeId)->update(['status' => 0]);
                Employee::find($employeeId)?->update(['status' => 1]);

                EmployeeClientChannel::create([
                    'department_id' => $request->department_id,
                    'employee_id' => $employeeId,
                    'store_id' => $request->storeID,
                    'position' => $request->position,
                    'date_hired' => $request->date_hired,
                    'employment_start_date' => $request->employment_start_date,
                    'employment_end_date' => $request->employment_end_date,
                    'salary' => $request->salary,
                    'status' => 1,
                    'entry_time' => $request->entry_time,
                    'default_rate_id' => $request->default_rate_id,
                    'exit_time' => $request->exit_time,
                ]);
            }

            $department = Department::find($request->department_id);

            return redirect()->back()->with('success', 'Employees successfully assigned to ' . $department->department_name . ' department!');
        } catch (ValidationException $e) {
            return back()->withErrors($e->validator)->withInput();
        } catch (\Exception $e) {
            Log::error($e);
            return back()->with('error', $e->getMessage())->withInput();
        }
    }

}
