<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\Agency;
use App\Models\User;
use App\Models\Client;
use Yajra\DataTables\DataTables;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;

class CompanyController extends Controller
{
    public function index(Request $request)
    {
        $privileges = $request->get('privileges');

        // Retrieve the authenticated user (if the user is logged in)
        $user = Auth::user();

        // Get the agency IDs the user is assigned to
        $agencyIds = $user->viewable_agency_ids;

        // Start building the query for Company, eager load the related 'agency' and 'user'
        $query = Company::with('agency.user', 'user');

        if (!empty($agencyIds)) {

            $query->whereHas('agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });
        }


        if ($request->ajax()) {

            $data = $query->get();

            // Return DataTables response with the data
            return DataTables::of($data)
                ->addColumn('action', function ($row) use ($privileges) {
                    $actionBtn = '<div class="flex items-center justify-center gap-2">';

                    if (in_array('View Company', $privileges)) {
                        $actionBtn .= '<a href="' . url('admin/company-profile/' . $row->id) . '" class="flex items-center justify-center w-10 h-10 text-white bg-green-600 rounded-md edit"><i class="text-xl bx bx-show"></i></a>';
                    }

                    if (in_array('Edit Company', $privileges)) {
                        $actionBtn .= '<button onclick="openModal(\'updateModal' . $row->id . '\')" class="flex items-center justify-center w-10 h-10 text-white bg-blue-600 rounded-md edit"><i class="bx bx-edit-alt"></i></button>';
                    }

                    if (in_array('Delete Company', $privileges)) {
                        $actionBtn .= '<button onclick="openModal(\'deleteModal' . $row->id . '\')" class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md delete"><i class="text-xl bx bx-trash-alt"></i></button>';
                    }

                    $actionBtn .= '</div>';
                    return $actionBtn;
                })
                ->make(true);
        }

        // For non-AJAX requests, get the agencies and companies data
        $agencies = Agency::all();
        $data = $query->get(); // Use the query to get the filtered company data

        // Return the view with agencies and company data
        return view('Admin.Company.index', compact('agencies', 'data'));
    }

    public function showcompanydetails(Request $request, $id)
    {
        if ($request->ajax()) {
            $data = Client::with('user')->where('company_id', $id)->get();
            return DataTables::of($data)
                ->addColumn('action', function ($row) {
                    $actionBtn = '<div class="flex items-center justify-center gap-2">
                    <a href="' . url('admin/client-profile/' . $row->id) . '" class="flex items-center justify-center w-10 h-10 text-white bg-green-600 rounded-md edit"><i class="text-xl bx bx-show"></i></a>
                    <button class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-m delete"><i class="text-xl bx bx-trash-alt" ></i></button>
                </div>';
                    return $actionBtn;
                })
                ->make(true);
        }
        $company = Company::with('user', 'agency')->find($id);
        return view('Admin.Company.companyprofile', compact('company'));
    }



    public function store(Request $request)
    {

        $request->validate([
            'agency_id' => 'required|exists:agencies,id',
            'name' => 'required|string|max:255',
            'contact_person' => 'required|string|max:255',
            'email' => 'required|string|email|max:255',
            'password' => 'required|string|min:8|confirmed',
            'contact_phone' => 'required|string|max:20',
            'address' => 'required|string|max:255',
            'industry' => 'required|string|max:255',
        ]);


        $user = User::create([
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role_id' => 2, // Assign role if necessary
        ]);


        Company::create([
            'user_id' => $user->id,
            'agency_id' => $request->agency_id,
            'name' => $request->name,
            'contact_person' => $request->contact_person,
            'contact_phone' => $request->contact_phone,
            'address' => $request->address,
            'industry' => $request->industry,
        ]);

        // Redirect with success message
        return redirect()->route('companies.index')->with('success', 'Company created successfully.');
    }

    public function show(Company $company)
    {
        return view('companies.show', compact('company'));
    }

    public function edit(Company $company)
    {
        $agencies = Agency::all();
        return view('companies.edit', compact('company', 'agencies'));
    }

    public function update(Request $request, Company $company)
    {
        // Validate request data
        $request->validate([
            'name' => 'required|string|max:255',
            'contact_person' => 'required|string|max:255',
            'contact_phone' => 'required|string|max:20',
            'address' => 'required|string|max:255',
        ]);

        // Update the agency with the validated data
        $company->update($request->only(['name', 'contact_person', 'contact_phone', 'address']));

        // Redirect to the agencies index with a success message
        return redirect()->route('companies.index')->with('success', $company->name . ' Company updated successfully.');
    }


    public function destroy(Company $company)
    {
        $company->delete();
        return redirect()->route('companies.index')->with('success', 'Company deleted successfully.');
    }
}
