<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Ptsr;
use App\Models\DepartmentStore;
use App\Models\Assign;
use App\Models\TrxCode;
use App\Models\Employee;
use App\Models\Attendance;
use App\Models\Department;
use App\Models\WorkSchedule;
use Illuminate\Http\Request;
use App\Models\AssignCompany;
use App\Imports\DeductionImport;
use Yajra\DataTables\DataTables;
use App\Imports\AttendanceImport;
use App\Imports\AttendanceImport2;
use App\Imports\AttendanceImport3;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\EmployeeClientChannel;
use App\Models\AssignCompanyDepartment;
use App\Models\Stores;
use ZipArchive;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use App\Events\AttendanceUpdated;
use Barryvdh\DomPDF\Facade\Pdf;
class AttendanceController extends Controller
{

    public function exportPdf(Request $request)
    {
        $departmentId = $request->department_id;
        $startDate = $request->startDate;
        $endDate = $request->endDate;

        // Get all employees in the department
        $employees = EmployeeClientChannel::with('employee', 'department')->where('department_id', $departmentId)->get();

        // If no employees found, return error
        if ($employees->isEmpty()) {
            return back()->with('error', 'No employees found in this department.');
        }

        // Temporary folder to store PDFs
        $tempPath = storage_path('app/public/temp_pdfs');
        if (!file_exists($tempPath)) {
            mkdir($tempPath, 0777, true);
        }

        $pdfFiles = [];

        foreach ($employees as $employee) {
            $query = Attendance::where('employee_id', $employee->employee_id)
                ->where('department_id', $departmentId);

            if ($startDate) {
                $query->whereDate('date', '>=', $startDate);
            }
            if ($endDate) {
                $query->whereDate('date', '<=', $endDate);
            }

            $attendances = $query->orderBy('date')->get();

            // Skip if no attendance
            if ($attendances->isEmpty()) {
                continue;
            }

            // Get employee's full name safely
            $fullName = trim(
                ($employee->employee->first_name ?? '') . ' ' .
                ($employee->employee->middle_name ?? '') . ' ' .
                ($employee->employee->last_name ?? '')
            );

            // If full name is empty, use employee ID as fallback
            if (empty($fullName)) {
                $fullName = 'employee_' . $employee->employee_id;
            }

            // Replace spaces and unsafe characters for file name
            $safeName = preg_replace('/[^A-Za-z0-9_-]/', '_', $fullName);

            $pdf = PDF::loadView(
                'Admin.Attendance.complete.dtr_report',
                compact('attendances', 'employee', 'startDate', 'endDate')
            );

            $pdfFileName = 'attendance_' . $safeName . '.pdf';
            $pdfFilePath = $tempPath . '/' . $pdfFileName;
            $pdf->save($pdfFilePath);

            $pdfFiles[] = $pdfFilePath;
        }

        // If more than 1 PDF, zip them
        if (count($pdfFiles) > 1) {
            // Get department name safely
            $departmentName = $employees->first()->department->department_name ?? 'department';
            $safeDeptName = preg_replace('/[^A-Za-z0-9_-]/', '_', $departmentName);

            $zipFileName = 'attendance_reports_' . $safeDeptName . '.zip';
            $zipFilePath = $tempPath . '/' . $zipFileName;

            $zip = new ZipArchive;
            if ($zip->open($zipFilePath, ZipArchive::CREATE | ZipArchive::OVERWRITE) === true) {
                foreach ($pdfFiles as $file) {
                    $zip->addFile($file, basename($file));
                }
                $zip->close();
            }

            // Clean up individual PDFs
            foreach ($pdfFiles as $file) {
                unlink($file);
            }

            return response()->download($zipFilePath)->deleteFileAfterSend(true);
        }

        // Only 1 PDF
        return response()->download($pdfFiles[0])->deleteFileAfterSend(true);
    }
    public function summaryattendance(Request $request, $employee_id, $department_id)
    {

        $user = Auth::user();

        // Get the agency IDs the user is assigned to
        $agencyIds = $user->viewable_agency_ids;


        $query = Attendance::with(['employee.channelclient'])->whereNull('status')->whereNull('approve')
            ->where('employee_id', $employee_id)
            ->where('department_id', $department_id);

        // Apply agency filter only if agency IDs exist
        if (!empty($agencyIds)) {
            $query->whereHas('employee.channelclient.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });
        }




        if ($request->ajax()) {
            $data = $query->get();


            return DataTables::of($data)

                ->addColumn('action', function ($row) {
                    $approveButton = '';

                    // Only show approve button if both time_in and time_out are NOT null
                    if (!is_null($row->time_in) && !is_null($row->time_out)) {
                        $approveButton = '
                        <button onclick="openApproveModal(' . $row->id . ')" 
                            class="flex items-center justify-center w-10 h-10 text-white bg-green-600 rounded-md">
                            <i class="text-xl bx bx-check"></i>
                        </button>';
                    }

                    return '
                    <div class="flex items-center justify-center gap-2">
                        ' . $approveButton . '
                        <button onclick="openDeleteModal(' . $row->id . ')" 
                            class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md">
                            <i class="text-xl bx bx-trash-alt"></i>
                        </button>
                    </div>';
                })


                ->addColumn('hours_worked', function ($row) {
                    if (!is_null($row->time_in) && !is_null($row->time_out)) {
                        $timeIn = \Carbon\Carbon::parse($row->time_in);
                        $timeOut = \Carbon\Carbon::parse($row->time_out);

                        // Calculate the difference in hours and minutes
                        $diff = $timeIn->diff($timeOut);
                        $hours = $diff->h;
                        $minutes = $diff->i;

                        return "{$hours} hrs {$minutes} mins";
                    }
                    return 'N/A';
                })

                ->addColumn('client_name', function ($row) {

                    if ($row->department_id != null) {
                        $dep = Department::find($row->department_id);

                        $client = $dep->department_name;
                    } else {
                        $client = 'No Client Associatated';
                    }

                    return $client;
                })
                ->addColumn('selfie21', function ($row) {
                    $imageUrl = $row->selfie ? Storage::disk('spaces')->url($row->selfie) : null;

                    if (!$imageUrl) {
                        return '<span class="text-gray-400">No selfie</span>';
                    }

                    return '
        <div class="flex items-center justify-center">
            <a href="' . $imageUrl . '" target="_blank" rel="noopener noreferrer">
                <img src="' . $imageUrl . '" alt="Selfie" width="60" height="60" class="cursor-pointer rounded shadow" />
            </a>
        </div>
    ';
                })

                ->addColumn('status', function ($row) {
                    return '
                        <input type="checkbox" class="status-checkbox" data-id="' . $row->id . '" 
                            ' . ($row->status == 1 ? 'checked' : '') . ' 
                            title="Toggle status">
                    ';
                })
                ->rawColumns(['status', 'selfie21', 'action'])
                ->make(true);
        }

        $employee = Employee::find($employee_id);
        $department = Department::find($department_id);

        $dtr = EmployeeClientChannel::where('employee_id', $employee_id)
            ->where('department_id', $department_id)->where('status', 1)->first();

        return view('Admin.Attendance.complete.summary', compact('employee', 'dtr', 'department'));
    }
    public function approveAttendance(Request $request)
    {
        if ($request->has('id')) { // Ensure 'id' exists in the request
            $attendance = Attendance::find($request->id);

            if (!$attendance) {
                return response()->json(['message' => 'Attendance record not found'], 404);
            }

            $attendance->approve = 1; // Update the approve column to 1
            $attendance->save();

            return response()->json(['message' => 'Attendance approved successfully'], 200);
        } else {
            Attendance::where('employee_id', $request->employee_id)
                ->where('department_id', $request->department_id)
                ->whereNull('status')
                ->whereNotNull('time_in')
                ->whereNotNull('time_out')
                ->whereNull('approve')
                ->update(['approve' => 1]);
        }

        return response()->json(['message' => 'Attendance approved successfully']);
    }
    public function updateAttendance(Request $request)
    {
        $attendance = Attendance::find($request->id);

        if (!$attendance) {
            return response()->json(['error' => 'Attendance record not found'], 404);
        }

        $attendance->{$request->field} = $request->value;
        $attendance->save();

        return response()->json(['message' => 'Attendance updated successfully']);
    }

    public function employeefordtr(Request $request)
    {

        $user = Auth::user();

        // Get the agency IDs the user is assigned to
        $agencyIds = $user->viewable_agency_ids;


        $query = EmployeeClientChannel::with('employee')
            ->whereHas('employee.attendance', function ($q) {
                $q->whereNull('status')
                    ->whereNull('approve');
            });

        if (!empty($agencyIds)) {

            $query->whereHas('department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });



            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            // Get the company IDs restricted via assignCompanies
            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', $restrictedCompanyIds)
                ->pluck('department_id')
                ->unique();


            // Apply company restriction only if there are any
            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereIn('department_id', $restrictedDepartmentIds);
            }
        }




        if ($request->ajax()) {
            $data = $query->get();


            return DataTables::of($data)
                ->addColumn('action', function ($row) {
                    $url = route('summaryattendance', ['employee_id' => $row->employee_id, 'department_id' => $row->department_id]);

                    return '
                    <div class="flex items-center justify-center gap-2">
                        <a href="' . $url . '" class="text-blue-500 hover:text-blue-700" title="View Summary">
                            <i class="text-xl bx bx-show"></i> <!-- Boxicon Eye Icon -->
                        </a>
                    </div>';
                })
                ->addColumn('period', function ($row) {
                    $attendancefirst = Attendance::where('employee_id', $row->employee_id)
                        ->whereNull('status')
                        ->whereNull('approve')
                        ->orderBy('date', 'asc')
                        ->first();

                    $attendancelast = Attendance::where('employee_id', $row->employee_id)
                        ->whereNull('status')
                        ->whereNull('approve')
                        ->orderBy('date', 'desc')
                        ->first();

                    $firstDate = $attendancefirst ? $attendancefirst->date : 'N/A';
                    $lastDate = $attendancelast ? $attendancelast->date : 'N/A';

                    return $firstDate . ' - ' . $lastDate;
                })

                ->addColumn('client_name', function ($row) {

                    if ($row->department_id != null) {
                        $dep = Department::find($row->department_id);

                        $client = $dep->department_name;
                    } else {
                        $client = 'No Client Associatated';
                    }

                    return $client;
                })
                ->addColumn('status', function ($row) {
                    return '
                         <input type="checkbox" class="status-checkbox" data-id="' . $row->id . '" 
                             ' . ($row->status == 1 ? 'checked' : '') . ' 
                             title="Toggle status">
                     ';
                })
                ->rawColumns(['status', 'action'])
                ->make(true);
        }
    }






    public function index(Request $request)
    {

        $user = Auth::user();

        // Get the agency IDs the user is assigned to
        $agencyIds = $user->viewable_agency_ids;


        $query = Attendance::with('employee.channelclient')->whereNotNull('approve');


        if (!empty($agencyIds)) {

            $query->whereHas('employee.channelclient.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });


            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            // Get the company IDs restricted via assignCompanies
            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', $restrictedCompanyIds)
                ->pluck('department_id')
                ->unique();


            // Apply company restriction only if there are any
            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereIn('department_id', $restrictedDepartmentIds);
            }
        }




        if ($request->ajax()) {
            $data = $query->get();


            return DataTables::of($data)
                ->addColumn('employee_name', function ($row) {
                    return ($row->employee->first_name ?? '') . ' ' . ($row->employee->middle_name ?? '') . ' ' . $row->employee->last_name;
                })
                ->addColumn('action', function ($row) {
                    return '
                <div class="flex items-center justify-center gap-2">
                    <button onclick="openEditModal(' . $row->id . ')" class="flex items-center justify-center w-10 h-10 text-white bg-blue-600 rounded-md">
                        <i class="bx bx-edit-alt"></i>
                    </button>
                    <button onclick="openDeleteModal(' . $row->id . ')" class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md">
                        <i class="text-xl bx bx-trash-alt"></i>
                    </button>
                </div>';
                })

                ->addColumn('client_name', function ($row) {

                    if ($row->department_id != null) {
                        $dep = Department::find($row->department_id);

                        $client = $dep->department_name;
                    } else {
                        $client = 'No Client Associatated';
                    }

                    return $client;
                })
                ->addColumn('status', function ($row) {
                    return '
                        <input type="checkbox" class="status-checkbox" data-id="' . $row->id . '" 
                            ' . ($row->status == 1 ? 'checked' : '') . ' 
                            title="Toggle status">
                    ';
                })
                ->rawColumns(['status', 'action'])
                ->make(true);
        }


        $query = Employee::whereHas('channelclient', function ($query) {
            $query->whereNotNull('department_id');
        }); // Remove the incorrect query() method

        // Apply additional filtering if $agencyIds is not empty

        if (!empty($agencyIds)) {
            $query->whereHas('channelclient.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });


            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id'))
                ->pluck('department_id')
                ->unique();

            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereHas('channelclient', function ($q) use ($restrictedDepartmentIds) {
                    $q->whereIn('department_id', $restrictedDepartmentIds);
                });
            }
        }



        $employees = $query->get();



        $querydepartment = Department::with('client'); // Define $query before using it


        if (!empty($agencyIds)) {
            $querydepartment->whereHas('client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id'))
                ->pluck('department_id')
                ->unique();

            if ($restrictedDepartmentIds->isNotEmpty()) {
                $querydepartment->whereIn('id', $restrictedDepartmentIds);

            }
        }



        $client = $querydepartment->get(); // Execute the query after applying conditions




        $data = Attendance::with('employee')->get();
        $stores = Stores::get();
        return view('Admin.Attendance.complete.index', compact('data', 'stores', 'client', 'employees'));
    }
    public function updateStatusToNull(Request $request, $id)
    {
        $attendance = Attendance::findOrFail($id);
        $attendance->status = $request->status == 1 ? 1 : null;
        $attendance->save();

        return response()->json([
            'success' => true,
            'message' => $attendance->status ? 'Status updated to 1 (Excluded from calculation).' : 'Status updated to NULL successfully.'
        ]);
    }


    public function updateStatusByDateRange(Request $request)
    {
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
        ]);

        // Update all records within the selected date range
        Attendance::whereBetween('date', [$request->start_date, $request->end_date])
            ->update(['status' => null]); // Change to 1 if you want to enable status instead

        return response()->json([
            'success' => true,
            'message' => "Attendance status updated successfully for the selected date range."
        ]);
    }






    public function overtime(Request $request)
    {
        $user = Auth::user();

        // Get the agency IDs the user is assigned to
        $agencyIds = $user->viewable_agency_ids;

        $query = Attendance::with('employee.channelclient')
            ->where('approve', 1);

        if (!empty($agencyIds)) {
            $query->whereHas('employee.channelclient.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });


            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            // Get the company IDs restricted via assignCompanies
            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', $restrictedCompanyIds)
                ->pluck('department_id')
                ->unique();


            // Apply company restriction only if there are any
            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereIn('department_id', $restrictedDepartmentIds);
            }
        }


        if ($request->ajax()) {
            $data = $query->get();

            return DataTables::of($data)
                ->addColumn('overtime', function ($row) {
                    $overtime = 0;

                    $attendanceDate = Carbon::parse($row->date)->format('Y-m-d');
                    $exitTime = null;

                    // Load the employee data to check if shift-based
                    $employee = Employee::find($row->employee_id);

                    if ($employee && $employee->is_shift == 1) {
                        // Employee uses shift-based scheduling
                        $workSchedule = WorkSchedule::where('employee_id', $row->employee_id)
                            ->whereDate('start_date', '<=', $row->date)
                            ->whereDate('end_date', '>=', $row->date)
                            ->with('shift')
                            ->first();

                        if ($workSchedule && $workSchedule->shift) {
                            $exitTime = Carbon::parse($attendanceDate . ' ' . $workSchedule->shift->shift_end);
                        }
                    }

                    // If no shift or not shift-based, fallback to employee channel setup
                    if (!$exitTime) {
                        $employeeChannel = EmployeeClientChannel::where('employee_id', $row->employee_id)
                            ->where('status', '1')
                            ->first();

                        if ($employeeChannel && $employeeChannel->exit_time) {
                            $exitTime = Carbon::parse($attendanceDate . ' ' . $employeeChannel->exit_time);
                        }
                    }

                    // Only calculate if both time_out and exitTime are valid
                    if ($row->time_out && $exitTime) {
                        $timeOut = Carbon::parse($attendanceDate . ' ' . $row->time_out);

                        if ($timeOut->gt($exitTime)) {
                            $overtimeDuration = $timeOut->diffInMinutes($exitTime);
                            $overtime = $overtimeDuration / 60;
                        }
                    }

                    return number_format($overtime, 2) . ' hours';
                })
                ->addColumn('employee_name', function ($row) {
                    return ($row->employee->first_name ?? '') . ' ' . ($row->employee->middle_name ?? '') . ' ' . $row->employee->last_name;
                })
                ->addColumn('late', function ($row) {
                    $late = 0;
                    $undertime = 0;

                    $attendanceDate = Carbon::parse($row->date)->format('Y-m-d');
                    $startTime = null;
                    $breakInTime = null;

                    // Load employee to check if shift-based
                    $employee = \App\Models\Employee::find($row->employee_id);

                    if ($employee && $employee->is_shift == 1) {
                        // Use shift schedule if is_shift is 1
                        $workSchedule = \App\Models\WorkSchedule::where('employee_id', $row->employee_id)
                            ->whereDate('start_date', '<=', $row->date)
                            ->whereDate('end_date', '>=', $row->date)
                            ->with('shift')
                            ->first();

                        if ($workSchedule && $workSchedule->shift) {
                            $startTime = Carbon::parse($attendanceDate . ' ' . $workSchedule->shift->shift_start);
                            $breakInTime = Carbon::parse($attendanceDate . ' ' . $workSchedule->shift->break_in);
                        }
                    }

                    // Fallback to channel entry_time if not using shift
                    if (!$startTime) {
                        $employeeChannel = \App\Models\EmployeeClientChannel::where('employee_id', $row->employee_id)
                            ->where('status', '1')
                            ->first();

                        if ($employeeChannel && $employeeChannel->entry_time) {
                            $startTime = Carbon::parse($attendanceDate . ' ' . $employeeChannel->entry_time);
                        }
                    }

                    // Calculate late if time_in exceeds shift start or break_in exceeds shift break start
                    if ($row->time_in && $startTime) {
                        $timeIn = Carbon::parse($attendanceDate . ' ' . $row->time_in);

                        // Check if time_in is late (exceeds the shift start time)
                        if ($timeIn->gt($startTime)) {
                            $late = $timeIn->diffInMinutes($startTime) / 60;
                        }

                        // If break_in time exceeds break_start (break-out time)
                        if ($breakInTime && $row->break_in) {
                            $breakIn = Carbon::parse($attendanceDate . ' ' . $row->break_in);

                            if ($breakIn->gt($breakInTime)) {
                                $late += $breakIn->diffInMinutes($breakInTime) / 60; // Add late time to total late
                            }
                        }
                    }

                    // If time_in is earlier than the shift start, calculate undertime
                    if ($row->time_in && $startTime && Carbon::parse($row->time_in)->lt($startTime)) {
                        $undertime = $startTime->diffInMinutes(Carbon::parse($row->time_in)) / 60;
                    }

                    // Combine late and undertime
                    $totalLateAndUndertime = $late + $undertime;

                    return number_format($totalLateAndUndertime, 2) . ' hours';
                })


                ->addColumn('action', function ($row) {
                    if ($row->ot === null) { // Check if status is null (Pending)
                        $actionBtns = '
                            <div class="flex justify-center gap-2">
                                <button onclick="changeStatus(' . $row->id . ', 1)" class="flex items-center justify-center w-32 h-10 text-white bg-green-600 rounded-md">
                                    <i class="bx bx-check"></i> Approve
                                </button>
                                <button onclick="changeStatus(' . $row->id . ', 0)" class="flex items-center justify-center w-32 h-10 text-white bg-red-600 rounded-md">
                                    <i class="bx bx-x"></i> Decline
                                </button>
                            </div>';
                    } else {
                        $statusText = $row->ot == 1 ? 'Approved' : 'Declined';
                        $actionBtns = '
                            <div class="flex justify-center">
                                <span class="px-4 py-2 rounded-md">' . $statusText . '</span>
                            </div>';
                    }
                    return $actionBtns;
                })


                ->make(true);
        }

        $employees = Employee::all();
        $data = Attendance::with('employee')->get();
        $client = Department::get();
        return view('Admin.Attendance.overtime.index', compact('data', 'employees', 'client'));
    }

    public function updateStatus(Request $request)
    {
        $attendance = Attendance::find($request->id);

        if ($attendance) {
            $attendance->ot = $request->status; // Update the status (1 for Approve, 0 for Decline)
            $attendance->save();

            $action = $request->status == 1 ? 'approved' : 'declined';
            return response()->json(['success' => true, 'message' => "Attendance record has been $action."]);
        }

        return response()->json(['success' => false, 'message' => 'Attendance record not found.']);
    }

    public function import2(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,csv',
            'department_id' => 'required|exists:department,id',
        ]);

        $import = new AttendanceImport2($request->department_id);
        Excel::import($import, $request->file('file'));

        return response()->json([
            'success' => true,
            'message' => 'Attendance imported successfully.',
            'failedRows' => $import->failedRows ?? [],
        ]);
    }

    public function import3(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,csv',
            'department_id' => 'required|exists:department,id',
        ]);

        $import = new AttendanceImport3($request->department_id);
        Excel::import($import, $request->file('file'));

        return response()->json([
            'success' => true,
            'message' => 'Attendance imported successfully.',
            'failedRows1' => $import->failedRows ?? [],
        ]);
    }




    public function em_attendance(Request $request, $id)
    {
        if ($request->ajax()) {
            $data = Attendance::with('employee')

                ->where('employee_id', $id)
                ->get();


            return DataTables::of($data)

                ->addColumn('approve', function ($row) {
                    if ($row->approve == 1) {
                        return '<span class="text-green-500">Approved</span>';
                    } else {
                        return '<span class="text-green-500">Pending</span>';
                    }
                })



                ->rawColumns(['approve', 'action'])

                ->make(true);
        }
    }




    public function index1(Request $request)
    {



        $user = Auth::user();

        // Get the agency IDs the user is assigned to
        $agencyIds = $user->viewable_agency_ids;


        $query = Attendance::with('employee')
            ->whereNull('time_in')
            ->orWhereNull('time_out');


        if (!empty($agencyIds)) {

            $query->whereHas('employee.channelclient.department.client.company.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });
        }




        if ($request->ajax()) {
            $data = $query->get();

            return DataTables::of($data)
                ->addColumn('action', function ($row) {
                    $actionBtn = '<div class="flex items-center justify-center gap-2">
                          <button onclick="openModal(\'updateModal' . $row->id . '\')" class="flex items-center justify-center w-10 h-10 text-white bg-blue-600 rounded-md"><i class="bx bx-edit-alt"></i></button>
                        <button onclick="openModal(\'deleteModal' . $row->id . '\')" class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md"><i class="text-xl bx bx-trash-alt"></i></button>
                    </div>';
                    return $actionBtn;
                })
                ->make(true);
        }

        $employees = Employee::all();
        $data = Attendance::with('employee')->get();
        return view('Admin.Attendance.incomplete.index', compact('data', 'employees'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'department_id' => 'required|exists:department,id',
            'date' => 'required|date',
            'time_in' => 'required|date_format:H:i',
            'time_out' => 'required|date_format:H:i|after:time_in',
            'notes' => 'nullable|string|max:255',
        ]);

        // Combine date with time_in and time_out
        $date = Carbon::parse($request->date);
        $timeIn = Carbon::parse($request->date . ' ' . $request->time_in);
        $timeOut = Carbon::parse($request->date . ' ' . $request->time_out);

        $attendance = new Attendance();
        $attendance->employee_id = $request->employee_id;
        $attendance->department_id = $request->department_id;
        $attendance->date = $date->toDateString();
        $attendance->time_in = $timeIn;
        $attendance->time_out = $timeOut;
        $attendance->notes = $request->notes;
        $attendance->status = null; // Default status
        $attendance->approve = null; // Default not yet approved

        $attendance->save();

        return redirect()->back()->with('success', 'Attendance record created successfully.');
    }




    public function attendacestore(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|string|max:255',
            'date' => 'required|date',
            'time_type' => 'required|string|in:time_in,time_out,break_out,break_in,ot_in,ot_out',
            'selfie' => 'required|image|max:2048'
        ]);

        try {
            $employeeId = $request->employee_id;

            $isAssigned = EmployeeClientChannel::where('status', 1)->find($employeeId);
            if (!$isAssigned) {
                return response()->json([
                    'success' => false,
                    'message' => 'Assigned employee not found.',
                ], 404);
            }

            $user = auth()->user();

            if ($user->role_id == 8) {
                $authorized = Department::where('user_id', $user->id)
                    ->where('id', $isAssigned->department_id)
                    ->exists();
            } elseif ($user->role_id == 23) {
                $authorized = Stores::where('user_id', $user->id)
                    ->where('id', $isAssigned->store_id)
                    ->exists();
            } else {
                $authorized = true;
            }

            if (!$authorized) {
                return response()->json([
                    'success' => false,
                    'message' => 'You are not authorized to log attendance for this store.',
                ], 403);
            }


            // Upload selfie
            // $picturePath = null;
            // if ($request->hasFile('selfie')) {
            //     $file = $request->file('selfie');
            //     $file_name = time() . '.' . $file->getClientOriginalExtension();
            //     $destination = public_path('selfie');
            //     $file->move($destination, $file_name);
            //     $picturePath = 'selfie/' . $file_name;
            // }
            $picturePath = null;
            if ($request->hasFile('selfie')) {
                $file = $request->file('selfie');
                $file_name = 'selfie/' . time() . '.' . $file->getClientOriginalExtension();
                Storage::disk('spaces')->put($file_name, file_get_contents($file), 'public');
                $picturePath = $file_name;
            }



            $serverTime = now();
            $timeType = $request->time_type;

            // For non-consolidation types
            if (!in_array($timeType, ['time_out', 'ot_out'])) {
                Attendance::create([
                    'employee_id' => $isAssigned->employee_id,
                    'department_id' => $isAssigned->department_id,
                    'date' => $request->date,
                    $timeType => $serverTime,
                    'selfie' => $picturePath,
                    'created_at' => $serverTime,
                    'updated_at' => $serverTime,
                ]);

                return response()->json([
                    'success' => true,
                    'message' => ucfirst(str_replace('_', ' ', $timeType)) . ' recorded successfully.'
                ]);
            }

            // Consolidation for time_out / ot_out
            $allLogs = Attendance::where('employee_id', $isAssigned->employee_id)
                ->where('date', $request->date)
                ->orderBy('created_at')
                ->get();

            if ($allLogs->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'No attendance logs found for today.'
                ], 404);
            }

            // Consolidate (exclude current scan from consolidation)
            $previousLogs = $allLogs->filter(function ($entry) use ($timeType, $serverTime) {
                return !$entry->$timeType || $entry->$timeType != $serverTime;
            });

            $summaryData = [
                'employee_id' => $isAssigned->employee_id,
                'department_id' => $isAssigned->department_id,
                'date' => $request->date,
                'time_in' => $previousLogs->whereNotNull('time_in')->first()?->time_in,
                'break_out' => $previousLogs->whereNotNull('break_out')->last()?->break_out,
                'break_in' => $previousLogs->whereNotNull('break_in')->last()?->break_in,
                'ot_in' => $previousLogs->whereNotNull('ot_in')->first()?->ot_in,
                'selfie' => $picturePath,
                'created_at' => $serverTime,
                'updated_at' => $serverTime,
            ];

            // Add the new scan time
            if ($timeType === 'time_out') {
                $summaryData['time_out'] = $serverTime;
                $summaryData['ot_out'] = $previousLogs->whereNotNull('ot_out')->last()?->ot_out;
            } else {
                $summaryData['ot_out'] = $serverTime;
                $summaryData['time_out'] = $previousLogs->whereNotNull('time_out')->last()?->time_out;
            }

            // Delete all non-break-only logs except the one just inserted
            Attendance::where('employee_id', $isAssigned->employee_id)
                ->where('date', $request->date)
                ->where(function ($query) {
                    $query->whereNotNull('time_in')
                        ->orWhereNotNull('time_out')
                        ->orWhereNotNull('ot_in')
                        ->orWhereNotNull('ot_out');
                })
                ->delete();

            // Save the summary
            Attendance::create($summaryData);

            return response()->json([
                'success' => true,
                'message' => 'Attendance summary saved successfully.'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Server error: ' . $e->getMessage()
            ], 500);
        }
    }



    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Attendance  $attendance
     * @return \Illuminate\Http\Response
     */
    // public function show(Attendance $attendance)
    // {
    //     return view('Admin.Attendance.complete.show', compact('attendance'));
    // }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Attendance  $attendance
     * @return \Illuminate\Http\Response
     */


    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Attendance  $attendance
     * @return \Illuminate\Http\Response
     */
    public function edit(Attendance $attendance)
    {
        return response()->json($attendance);
    }

    public function update(Request $request, $id)
    {
        // Validate the request input
        $request->validate([
            'employee_id' => 'required|integer|exists:employees,id',
            'department_id' => 'nullable',
            'date' => 'required|date',
            'time_in' => ['required', 'regex:/^\d{2}:\d{2}(:\d{2})?$/'],  // Accepts H:i and H:i:s
            'time_out' => ['required', 'regex:/^\d{2}:\d{2}(:\d{2})?$/'],
            'notes' => 'nullable|string|max:255',
        ]);

        // Find the attendance record
        $attendance = Attendance::findOrFail($id);

        // Update the attendance record
        $attendance->update($request->only(['employee_id', 'department_id', 'date', 'time_in', 'time_out', 'notes']));

        // Return response based on request type (AJAX or normal)
        if ($request->ajax()) {
            return response()->json(['success' => true, 'message' => 'Attendance updated successfully.']);
        }

        return redirect()->back()->with('success', 'Attendance updated successfully.');
    }

    public function destroy(Request $request, Attendance $attendance)
    {
        // Delete the selfie image file if it exists
        if ($attendance->selfie) {
            $filePath = public_path($attendance->selfie);
            if (file_exists($filePath)) {
                @unlink($filePath); // use @ to suppress errors if file already gone
            }
        }

        // Delete the attendance record
        $attendance->delete();

        // Return response
        if ($request->ajax()) {
            return response()->json(['success' => true, 'message' => 'Attendance and selfie deleted successfully.']);
        }

        return redirect()->back()->with('success', 'Attendance and selfie deleted successfully.');
    }



    public function import(Request $request)
    {
        // Validate the uploaded file
        $request->validate([
            'file' => 'required|mimes:xlsx,csv,xls',
        ]);

        // Import the attendance data
        Excel::import(new AttendanceImport, $request->file('file'));

        return redirect()->back()->with('success', 'Attendance data imported successfully.');
    }


    public function getClientsByEmployee(Request $request)
    {
        // Validate the incoming request to ensure 'employee_id' is provided and valid
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
        ]);

        // Fetch the clients associated with the selected employee, eager load 'department'
        $employeeClientChannels = EmployeeClientChannel::where('employee_id', $request->employee_id)
            ->with('department')->where('status', '1')
            ->get();

        // Check if the collection is empty
        if ($employeeClientChannels->isEmpty()) {
            return response()->json(['message' => 'No clients found for this employee'], 404);
        }

        // Return the clients as a JSON response
        return response()->json(
            $employeeClientChannels->map(function ($employeeClientChannel) {
                return [
                    'id' => optional($employeeClientChannel->department)->id,  // Safely access the department ID
                    'name' => optional($employeeClientChannel->department)->department_name,  // Corrected typo
                ];
            })
        );
    }


    public function deduction(Request $request, $status)
    {
        $user = Auth::user();

        // Get the agency IDs the user is assigned to
        $agencyIds = $user->viewable_agency_ids;


        $query = Ptsr::where('status', $status)->with('employee.channelclient')
            ->latest('created_at');


        if (!empty($agencyIds)) {

            $query->whereHas('employee.channelclient.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });


            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            // Get the company IDs restricted via assignCompanies
            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', $restrictedCompanyIds)
                ->pluck('department_id')
                ->unique();


            // Apply company restriction only if there are any
            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereIn('department_id', $restrictedDepartmentIds);
            }
        }







        $data = $query->get();

        if ($request->ajax()) {
            return DataTables::of($data)
                ->addColumn('s_date', function ($row) {
                    return $row->start_date;
                })
                ->addColumn('e_date', function ($row) {
                    return $row->end_date;
                })
                ->addColumn('action', function ($row) {
                    return '
                    <div class="flex items-center justify-center gap-2">
                        <a href="' . route('ptsr.show', $row->id) . '" class="flex items-center justify-center w-10 h-10 text-white bg-blue-600 rounded-md">
                            <i class="text-xl bx bx-show"></i>
                        </a>
                        <button onclick="openDeleteModal(' . $row->id . ')" class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md">
                            <i class="text-xl bx bx-trash-alt"></i>
                        </button>
                    </div>';
                })
                ->addColumn('department', function ($row) {
                    $department = Department::find($row->department_id); // Use department_id instead of row id
                    return $department ? $department->department_name : 'N/A'; // Handle null cases
                })



                ->addColumn('status', function ($row) {
                    return '
                        <input type="checkbox" class="status-checkbox" data-id="' . $row->id . '" 
                            ' . ($row->status2 == 1 ? 'checked' : '') . ' 
                            title="Toggle status">
                    ';
                })
                ->rawColumns(['status', 'action'])
                ->make(true);
        }

        $querydepartment = Department::with('client');

        if (!empty($agencyIds)) {

            $querydepartment->whereHas('client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });


            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            // Get the company IDs restricted via assignCompanies
            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', $restrictedCompanyIds)
                ->pluck('department_id')
                ->unique();


            // Apply company restriction only if there are any
            if ($restrictedDepartmentIds->isNotEmpty()) {
                $querydepartment->whereIn('id', $restrictedDepartmentIds);
            }
        }

        $client = $querydepartment->get(); // Execute the query after applying conditions

        $query = Employee::whereHas('channelclient', function ($query) {
            $query->whereNotNull('department_id');
        }); // Remove the incorrect query() method

        // Apply additional filtering if $agencyIds is not empty
        if (!empty($agencyIds)) {
            $query->whereHas('channelclient.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });
        }

        // Fetch the employees after all conditions are applied
        $employees = $query->get();


        $trxCodes = TrxCode::get();

        return view('Admin.Attendance.deduction_earnings.index', compact('client', 'employees', 'trxCodes'));
    }

    public function importdeduction(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,csv,xls',
            'department_id' => 'required',
            'start_date' => 'required|date',
            'end_date' => 'required|date',
            'status' => 'nullable',
        ]);

        try {
            $import = new DeductionImport(
                $request->input('start_date'),
                $request->input('end_date'),
                $request->input('department_id'),
                $request->input('status')
            );

            Excel::import($import, $request->file('file'));

            return response()->json([
                'success' => true,
                'message' => 'PTS Records imported successfully.',
                'failedRows' => $import->getFailedRows(),
            ]);
        } catch (\Exception $e) {
            Log::error('Error importing file: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error importing file. Please check your data format.',
                'error_details' => $e->getMessage(),
            ], 500);
        }
    }
}
