<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Events\AfterSheet;
use Illuminate\Support\Collection;

class PayrollReportExport implements FromCollection, WithHeadings, WithEvents, ShouldAutoSize
{
    protected $batchCode, $payrollData, $uniqueHeaders, $client;

    public function __construct($batchCode, $payrollData, $uniqueHeaders, $client)
    {
        $this->batchCode = $batchCode;
        $this->payrollData = $payrollData;
        $this->uniqueHeaders = $uniqueHeaders;
        $this->client = $client;
    }

    public function headings(): array
    {
        return [
            ['Employee', 'Rate', 'Earnings', 'Deductions', 'Net Pay'],
        ];
    }

    public function collection()
    {
        $rows = new Collection();

        foreach ($this->payrollData as $data) {
            $payroll = $data['payroll'];

            // ✅ Compute Basic Salary
            $basicSalary = $payroll->days_worked_withdeduction * $payroll->rate;

            // ✅ Compute total earnings (Basic + Allowances)
            $earnings = $basicSalary;
            foreach ($payroll->othersForPayroll as $other) {
                $earnings += $other->amount;
            }

            // Employee summary row
            $rows->push([
                $payroll->employee->last_name . ', ' . $payroll->employee->first_name,
                $payroll->rate,
                number_format($earnings, 2),
                number_format($payroll->deductions, 2),
                number_format($payroll->earnings, 2), // assumed Net Pay
            ]);

            $bankName = $payroll->employee->channelBank?->bankprofile?->name ?? 'No Bank';
            $bankAccount = $payroll->employee->channelBank?->bankaccount ?? 'No Account';

            $rows->push([
                $bankName . ' - ' . $bankAccount
            ]);

            $rows->push([
                'Gross Pay & Allowances',
                'Benefits',
                'Deductions',
                'Loans',
                '',
            ]);

            // Sub-values
            $maxRows = max(
                $payroll->othersForPayroll->count(),
                count($data['benefits']),
                $payroll->trxConstantDeductions->count() + count($data['nonStandardDeductions']),
                $payroll->trxLoanDeductions->count()
            );
            for ($i = 0; $i < $maxRows; $i++) {
                $rows->push([
                    // Gross & Allowances
                    $i === 0
                    ? 'Basic Salary: ' . number_format($basicSalary, 2)
                    : (isset($payroll->othersForPayroll[$i - 1])
                        ? $payroll->othersForPayroll[$i - 1]->name . ': ' . number_format($payroll->othersForPayroll[$i - 1]->amount, 2)
                        : ''),

                    // Benefits
                    isset($data['benefits'][$i])
                    ? $data['benefits'][$i]['type'] . ': ' . number_format($data['benefits'][$i]['amount'], 2)
                    : '',

                    // Deductions
                    isset($payroll->trxConstantDeductions[$i])
                    ? ($payroll->trxConstantDeductions[$i]->trxConstant->trxCode->description ?? 'Deduction')
                    . ': ' . number_format($payroll->trxConstantDeductions[$i]->E_Amount, 2)
                    : (isset($data['nonStandardDeductions'][$i])
                        ? $data['nonStandardDeductions'][$i]['type'] . ': ' . number_format($data['nonStandardDeductions'][$i]['amount'], 2)
                        : ''),

                    // Loans
                    isset($payroll->trxLoanDeductions[$i])
                    ? ($payroll->trxLoanDeductions[$i]->trxLoan->trxCode->description ?? 'Loan')
                    . ': ' . number_format($payroll->trxLoanDeductions[$i]->Amount, 2)
                    : '',

                    '',
                ]);
            }

            // Add a blank row (separator, no borders)
            $rows->push(['']);

        }

        $rows->push(['']);
        $rows->push(['']);
        $rows->push(['Transaction', 'Earnings', 'Deductions']);

        $earningsRows = [];
        $deductionRows = [];
        $totalEarnings = $totalDeductions = 0;

        foreach ($this->payrollData as $data) {
            $payroll = $data['payroll'];

            $basic = $payroll->days_worked_withdeduction * $payroll->rate;
            $earningsRows['Basic Salary'] = ($earningsRows['Basic Salary'] ?? 0) + $basic;
            $totalEarnings += $basic;

            foreach ($payroll->othersForPayroll as $other) {
                $earningsRows[$other->name] = ($earningsRows[$other->name] ?? 0) + $other->amount;
                $totalEarnings += $other->amount;
            }

            foreach ($data['benefits'] as $benefit) {
                $deductionRows[$benefit['type']] = ($deductionRows[$benefit['type']] ?? 0) + $benefit['amount'];
                $totalDeductions += $benefit['amount'];
            }

            foreach ($data['nonStandardDeductions'] as $ns) {
                $deductionRows[$ns['type']] = ($deductionRows[$ns['type']] ?? 0) + $ns['amount'];
                $totalDeductions += $ns['amount'];
            }

            foreach ($payroll->trxConstantDeductions as $deduction) {
                $desc = $deduction->trxConstant->trxCode->description ?? 'Unknown';
                if ($deduction->status == 1) {
                    $earningsRows[$desc] = ($earningsRows[$desc] ?? 0) + $deduction->E_Amount;
                    $totalEarnings += $deduction->E_Amount;
                } else {
                    $deductionRows[$desc] = ($deductionRows[$desc] ?? 0) + $deduction->E_Amount;
                    $totalDeductions += $deduction->E_Amount;
                }
            }

            foreach ($payroll->trxLoanDeductions as $loan) {
                $desc = $loan->trxLoan->trxCode->description ?? 'Loan';
                $deductionRows[$desc] = ($deductionRows[$desc] ?? 0) + $loan->Amount;
                $totalDeductions += $loan->Amount;
            }
        }

        ksort($earningsRows, SORT_NATURAL | SORT_FLAG_CASE);
        ksort($deductionRows, SORT_NATURAL | SORT_FLAG_CASE);

        foreach ($earningsRows as $name => $amount) {
            $rows->push([$name, $amount, '']);
        }
        foreach ($deductionRows as $name => $amount) {
            $rows->push([$name, '', $amount]);
        }

        $rows->push(['Total', $totalEarnings, $totalDeductions]);

        return $rows;
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function (AfterSheet $event) {
                $sheet = $event->sheet;

                // Header style
                $sheet->getStyle('A1:E1')->applyFromArray([
                    'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
                    'fill' => ['fillType' => 'solid', 'color' => ['rgb' => '1E3A8A']],
                    'alignment' => ['horizontal' => 'center', 'vertical' => 'center'],
                ]);

                $highestRow = $sheet->getHighestRow();

                // Apply borders only to non-blank rows
                for ($row = 2; $row <= $highestRow; $row++) {
                    $isBlank = empty(array_filter($sheet->getDelegate()->rangeToArray("A{$row}:E{$row}")[0]));
                    if (!$isBlank) {
                        $sheet->getStyle("A{$row}:E{$row}")->applyFromArray([
                            'borders' => [
                                'allBorders' => [
                                    'borderStyle' => 'thin',
                                    'color' => ['rgb' => 'E5E7EB'],
                                ],
                            ],
                        ]);
                        if (!empty($rowData[2])) { // column C holds earnings
                            $sheet->getStyle("A{$row}")->applyFromArray([
                                'font' => ['bold' => true],
                            ]);
                        }
                    }
                }

                // Column widths
                $sheet->getColumnDimension('A')->setWidth(32);
                $sheet->getColumnDimension('B')->setWidth(14);
                $sheet->getColumnDimension('C')->setWidth(22);
                $sheet->getColumnDimension('D')->setWidth(24);
                $sheet->getColumnDimension('E')->setWidth(18);
            },
        ];
    }
}
