<?php

namespace App\Exports;

use App\Models\Employee;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use Carbon\Carbon;

class EmployeeDepartmentExport implements FromCollection, WithHeadings, WithMapping, WithStyles, ShouldAutoSize
{
    protected $departmentId;
    protected $rowIndex = 0;

    public function __construct($departmentId = null)
    {
        $this->departmentId = $departmentId;
    }

    public function collection()
    {
        $query = Employee::with('channelclient.department');

        if (!empty($this->departmentId)) {
            $query->whereHas('channelclient', function ($q) {
                $q->where('department_id', $this->departmentId);
            });
        }

        return $query->get();
    }

    public function headings(): array
    {
        return [
            '#',
            'Employee Name',
            'Departments',
            'Date of Birth',
            'Age',
            'Present Address',
            'TIN No',
            'SSS No',
            'Pagibig No',
            'Philhealth No',
        ];
    }

    public function map($employee): array
    {
        $this->rowIndex++;

        $fullName = trim(
            $employee->last_name . ', ' . $employee->first_name . ' ' . ($employee->middle_name ?? '')
        );

        $departments = $employee->channelclient->map(function ($channel) {
            $deptName = $channel->department->department_name ?? 'Unknown Department';
            return $channel->status == 1 ? $deptName : $deptName . ' (Inactive)';
        })->implode(', ');

        return [
            $this->rowIndex,
            $fullName,
            $departments ?: 'Not assigned',
            $employee->date_of_birth ? Carbon::parse($employee->date_of_birth)->format('M d, Y') : '',
            $employee->date_of_birth ? Carbon::parse($employee->date_of_birth)->age : '',
            $employee->present_address ?? '',
            $employee->tin_no ?? '',
            $employee->sss_no ?? '',
            $employee->pagibig_no ?? '',
            $employee->philhealth_no ?? '',
        ];
    }

    public function styles(Worksheet $sheet)
    {
        // Style header row
        $sheet->getStyle('A1:J1')->applyFromArray([
            'font' => [
                'bold' => true,
                'color' => ['rgb' => 'FFFFFF'],
            ],
            'fill' => [
                'fillType' => 'solid',
                'color' => ['rgb' => '1E3A8A'], // Tailwind blue-900
            ],
            'alignment' => [
                'horizontal' => 'center',
                'vertical' => 'center',
            ],
        ]);

        // Borders for all cells
        $sheet->getStyle('A1:J' . $sheet->getHighestRow())->applyFromArray([
            'borders' => [
                'allBorders' => [
                    'borderStyle' => 'thin',
                    'color' => ['rgb' => 'E5E7EB'], // Tailwind gray-200
                ],
            ],
        ]);

        return [];
    }
}
